/*
 * Java
 *
 * Copyright 2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests kerning.
 */
public class TestFontComplexLayout {

	private static final String AMIRI_FONT = "/fonts/Amiri-reduced.ttf";
	private static final String NOTOSANSARABIC_FONT = "/fonts/NotoSansArabic-reduced.ttf";
	private static final String NOTOSERIFTHAI_FONT = "/fonts/NotoSerifThai-Regular.ttf";

	private static final String ARABIC_STRING = "الطريق أخضر ومظلل";
	private static final String THAI_STRING = "ผีเสื้อสีแดงกำลังโบยบิน";
	private static final String[] ARABIC_STRINGS = { "وكان معه المساع", "الذي يحمل له", "عدة التصوير،",
			"ويظهر في الصور", "كيف كان شكل الكاميرا" };

	private static final String[] THAI_STRINGS = { "ลิงกับแมวเหมือนช้าง", "ส่วนที่ดีที่สุดในชีวิตฉัน",
			"ลูกสาวของฉันเป็นหมอฟัน", "จะไปประเทศใครบ้าง", "หมาแดงอยู่ในสระว่ายน้ำ" };

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing an arabic string with 2 fonts with and without complex layout
	 *
	 */
	@Test
	public void testDrawStringLayoutArabic() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font0 = VectorFont.loadFont(AMIRI_FONT);
		VectorFont font1 = VectorFont.loadFont(NOTOSANSARABIC_FONT);
		VectorFont font2 = VectorFont.loadFont(AMIRI_FONT, true);
		VectorFont font3 = VectorFont.loadFont(NOTOSANSARABIC_FONT, true);

		String string = ARABIC_STRING;

		int fontSize = 30;
		int x = 50;

		// simple layout font0
		g.setColor(Colors.RED);
		int y = 50;
		VectorFont font = font0;

		int fontHeight = (int) font.getHeight(fontSize);
		checkFont(TestUtilities.getImageContextualName("testDrawStringLayoutArabic_amiri_simple"), g, font, string,
				fontSize, x, y);

		// simple layout font1

		g.setColor(Colors.WHITE);
		y += fontHeight + 10;
		font = font1;

		fontHeight = (int) font.getHeight(fontSize);
		checkFont(TestUtilities.getImageContextualName("testDrawStringLayoutArabic_notoarabic_simple"), g, font, string,
				fontSize, x, y);

		// complex layout font0
		g.setColor(Colors.GREEN);
		y += fontHeight + 10;
		font = font2;

		fontHeight = (int) font.getHeight(fontSize);
		checkFont(TestUtilities.getImageContextualName("testDrawStringLayoutArabic_amiri_complex"), g, font, string,
				fontSize, x, y);

		// complex layout font1
		g.setColor(Colors.YELLOW);
		y += fontHeight + 10;
		font = font3;

		fontHeight = (int) font.getHeight(fontSize);
		checkFont(TestUtilities.getImageContextualName("testDrawStringLayoutArabic_notoarabic_complex"), g, font,
				string, fontSize, x, y);

	}

	/**
	 * Test drawing a thai string with 1 font with and without complex layout
	 *
	 */
	@Test
	public void testDrawStringLayoutThai() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font0 = VectorFont.loadFont(NOTOSERIFTHAI_FONT);
		VectorFont font1 = VectorFont.loadFont(NOTOSERIFTHAI_FONT, true);

		String string = THAI_STRING;

		int fontSize = 30;
		int x = 30;

		// simple layout font0
		g.setColor(Colors.RED);
		int y = 50;
		VectorFont font = font0;
		int fontHeight = (int) font.getHeight(fontSize);
		checkFont(TestUtilities.getImageContextualName("testDrawStringLayoutThai_notothai_simple"), g, font, string,
				fontSize, x, y);

		// complex layout font1
		g.setColor(Colors.WHITE);
		y += fontHeight + 10;
		font = font1;

		fontHeight = (int) font.getHeight(fontSize);
		checkFont(TestUtilities.getImageContextualName("testDrawStringLayoutThai_notothai_complex"), g, font, string,
				fontSize, x, y);

	}

	/**
	 * Test drawing an arabic string with transformations
	 *
	 */
	@Test
	public void testDrawStringLayoutTransform() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);

		String string = ARABIC_STRING;

		int fontSize = 30;
		LinearGradient gradient = new LinearGradient(0, 0, 150, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		g.setColor(Colors.RED);
		Matrix m = new Matrix();

		m.setTranslate(60, 30);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, m, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		m.preTranslate(0, 40);
		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, m, gradient, 0xFF, BlendMode.SRC_OVER, 0);

		m.preTranslate(0, 40);
		m.preRotate(45);

		g.setColor(Colors.GREEN);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, m, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		m.preRotate(-90);
		m.postTranslate(100, 140);
		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, m, gradient, 0xFF, BlendMode.SRC_OVER, 0);

		g.setColor(Colors.WHITE);
		m.preRotate(45);
		m.postTranslate(-100, 80);

		m.preScale(1, -1);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, m, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		m.postTranslate(0, 40);
		m.preScale(1.5f, 1);
		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, m, gradient, 0xFF, BlendMode.SRC_OVER, 0);

		display.flush();

		float delta = TestUtilities.isOnAndroid() ? 0.10f : 0.05f;
		TestUtilities.compareDisplay(TestUtilities.getImageContextualName("testDrawStringLayoutTransform"),
				Colors.BLACK, 0, 0, display.getWidth(), display.getHeight(), delta);
	}

	/**
	 * Test drawing an arabic text
	 *
	 */
	@Test
	public void testDrawStringLayoutArabicText() {

		// Test disabled for now due to issue https://youtrack.cross/issue/M0092MEJAUI-2652
		if (TestUtilities.isOnAndroid()) {
			return;
		}

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);

		g.setColor(Colors.RED);

		int xRight = 300;
		Painter.drawLine(g, xRight, 0, xRight, display.getHeight());

		int fontSize = 30;
		g.setColor(Colors.WHITE);
		Matrix m = new Matrix();

		for (int i = 0; i < ARABIC_STRINGS.length; i++) {
			String string = ARABIC_STRINGS[i];
			assert string != null;

			float stringWidth = font.measureStringWidth(string, fontSize);
			int x = (int) (xRight - stringWidth);
			int y = (int) (40 + i * font.getHeight(fontSize));

			m.setTranslate(x, y);

			VectorGraphicsPainter.drawString(g, string, font, fontSize, m, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
					0);
		}

		display.flush();
		TestUtilities.compareDisplay(TestUtilities.getImageContextualName("testDrawStringLayoutArabicText"),
				Colors.BLACK, 0, 0, display.getWidth(), display.getHeight(), 0.05f);
	}

	/**
	 * Test drawing an thai text
	 *
	 */
	@Test
	public void testDrawStringLayoutThaiText() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSERIFTHAI_FONT, true);

		g.setColor(Colors.RED);

		int xLeft = 30;
		Painter.drawLine(g, xLeft, 0, xLeft, display.getHeight());

		int fontSize = 30;
		g.setColor(Colors.WHITE);
		Matrix m = new Matrix();

		for (int i = 0; i < THAI_STRINGS.length; i++) {
			String string = THAI_STRINGS[i];
			assert string != null;

			float stringWidth = font.measureStringWidth(string, fontSize);
			int x = (xLeft);
			int y = (int) (40 + i * font.getHeight(fontSize));

			m.setTranslate(x, y);

			VectorGraphicsPainter.drawString(g, string, font, fontSize, m, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
					0);
		}

		display.flush();

		float delta = TestUtilities.isOnAndroid() ? 0.15f : 0.05f;
		TestUtilities.compareDisplay(TestUtilities.getImageContextualName("testDrawStringLayoutThaiText"), Colors.BLACK,
				0, 0, display.getWidth(), display.getHeight(), delta);
	}

	private void checkFont(String testName, GraphicsContext g, VectorFont font, String string, float fontSize, int x,
			int y) {

		int width = (int) Math.ceil(font.measureStringWidth(string, fontSize));
		int height = (int) Math.ceil(font.getHeight(fontSize));
		drawRectangleCorners(g, x, y, width, height, 4);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		Display.getDisplay().flush();

		float delta = TestUtilities.isOnAndroid() ? 0.15f : 0.05f;
		TestUtilities.compareDisplay(testName, Colors.BLACK, x - 1, y - 1, width + 2, height + 2, delta);
	}

	private void drawRectangleCorners(GraphicsContext g, int x, int y, int width, int height, int size) {
		Painter.drawRectangle(g, x, y, size, size);
		Painter.drawRectangle(g, x + width - size, y, size, size);
		Painter.drawRectangle(g, x + width - size, y + height - size, size, size);
		Painter.drawRectangle(g, x, y + height - size, size, size);
	}

}
