/*
 * Java
 *
 * Copyright 2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.rcommand.comm;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import ej.ecom.io.CommConnection;
import ej.ecom.io.Connector;
import ej.rcommand.RemoteCommandManager;
import ej.rcommand.connectivity.ConnectivityListener;
import ej.rcommand.connectivity.ConnectivityManager;
import ej.rcommand.impl.StreamRemoteConnection;

/**
 * Adds a comm connection in the Rcommand manager.
 */
public class RemoteCommandCommPopulator implements ConnectivityManager {

	private final String connectionString;
	private final Logger logger;
	private CommConnection commConnection;
	private StreamRemoteConnection streamRemoteConnection;
	private RemoteCommandManager commandManager;
	private final List<ConnectivityListener> listeners;

	/**
	 * @param connectionString
	 *            the connection string of the comm connection to create.
	 */
	public RemoteCommandCommPopulator(String connectionString) {
		this.connectionString = connectionString;
		this.logger = Logger.getLogger(RemoteCommandCommPopulator.class.getSimpleName());
		this.listeners = new ArrayList<>();
	}

	/**
	 * Creates the comm connection and adds it in the given remote command
	 * manager.
	 *
	 * @param commandManager
	 *            the command manager to use to manage the created comm
	 *            connection.
	 * @throws IOException
	 *             if a I/O error occurs.
	 */
	public synchronized void start(RemoteCommandManager commandManager) throws IOException {
		this.commandManager = commandManager;
		this.commConnection = (CommConnection) Connector.open(this.connectionString);
		this.streamRemoteConnection = new StreamRemoteConnection(this.commConnection.openInputStream(),
				this.commConnection.openOutputStream());
		commandManager.startReading(this.streamRemoteConnection, this.connectionString);
		this.notifyConnectivityListener(true);
	}

	/**
	 * Stops the reading of the comm connection and closes it.
	 */
	public synchronized void stop() {
		if (this.commConnection != null) {
			this.notifyConnectivityListener(true);
			try {
				this.commandManager.stopReading(this.streamRemoteConnection);
				this.commConnection.close();
				this.commConnection = null;
			} catch (IOException e) {
				this.logger.log(Level.WARNING, "Exception thrown when closing the comm connection.", e);
			}
		}
	}

	@Override
	public synchronized int getAvailable() {
		return this.commConnection == null ? 0 : 1;
	}

	@Override
	public String getConnectionInfo() {
		if (this.commConnection != null) {
			return this.connectionString;
		} else {
			return null;
		}
	}

	private synchronized void notifyConnectivityListener(boolean connect) {
		if (connect) {
			for (ConnectivityListener listener : this.listeners) {
				listener.onConnect();
			}
		} else {
			for (ConnectivityListener listener : this.listeners) {
				listener.onDisconnect();
			}
		}
	}

	@Override
	public synchronized void addConnectivityListener(ConnectivityListener listener) {
		if (!this.listeners.contains(listener)) {
			this.listeners.add(listener);
		}
	}

	@Override
	public synchronized void removeConnectivityListener(ConnectivityListener listener) {
		this.listeners.remove(listener);
	}
}
