/*
 * Java
 *
 * Copyright 2021-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.net.test.util;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;

import ej.ecom.network.IPConfiguration;
import ej.ecom.network.NetworkInterfaceManager;

/**
 * Init service for Net testsuite using ethernet connection.
 */
public class EthernetNetInit implements INetInit {

	/* property used to set network interface name */
	public static final String NETIF_NAME_PROPERTY = "netif.name"; //$NON-NLS-1$

	/*
	 * property used to set the static ip address
	 * netif.ip=null => the address is provided by DHCP 
	 * netif.ip=xx.xx.xx.xx => the address provided by the user
	 */
	private static final String NETIF_CONF_PROP_IP = "netif.ip"; //$NON-NLS-1$

	/* property used to set the netmask (used only if netif.ip is set) */
	private static final String NETIF_CONF_PROP_NETMASK = "netif.netmask"; //$NON-NLS-1$
	
	/* property used to set the gateway (used only if netif.ip is set) */
	private static final String NETIF_CONF_PROP_GATEWAY = "netif.gateway"; //$NON-NLS-1$

	/* property used to set a static dns ip address */
	private static final String NETIF_CONF_PROP_DNS = "netif.dns"; //$NON-NLS-1$

	public static final String NETIF_NAME = System.getProperty(NETIF_NAME_PROPERTY);
	public static final String NETIF_IP = System.getProperty(NETIF_CONF_PROP_IP);
	public static final String NETIF_NETMASK = System.getProperty(NETIF_CONF_PROP_NETMASK);
	public static final String NETIF_GATEWAY = System.getProperty(NETIF_CONF_PROP_GATEWAY);
	public static final String NETIF_DNS = System.getProperty(NETIF_CONF_PROP_DNS);

	@Override
	public void initialize() throws IOException {

		if (NETIF_NAME == null) {
			throw new RuntimeException("[ERROR] No network interface defined. Please set the system property '" //$NON-NLS-1$
					+ NETIF_NAME_PROPERTY + "'."); //$NON-NLS-1$
		}
		startNetworkInterface(NETIF_NAME, NETIF_IP, NETIF_NETMASK, NETIF_GATEWAY, NETIF_DNS);
	}

	/**
	 * Start the network interface.
	 *
	 * @param interfaceName
	 *            the name of the network interface to start.
	 * @param staticIP
	 *            the static IP address to be used by the network interface. If <code>null</code>, dhcp configuration
	 *            will be used to get an IP address.
	 * @param netmask
	 *            the netmask to be used if <code>staticIP</code> if set.
	 * @param gateway
	 *            the gateway to be used if <code>staticIP</code> if set.
	 * @param dns
	 *            the dns ip to be used.
	 * @throws IOException
	 *             if an error occurs.
	 */
	public void startNetworkInterface(String interfaceName, String staticIP, String netmask, String gateway, String dns)
			throws IOException {

		NetworkInterface netif = NetworkInterface.getByName(interfaceName);
		if (netif == null) {
			throw new IOException("No network interface \"" + interfaceName + "\" found"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (!NetworkInterfaceManager.isStarted(netif)) {
			NetworkInterfaceManager.start(netif);
		}
		// force to enable network interface
		if (!NetworkInterfaceManager.isEnabled(netif)) {
			NetworkInterfaceManager.enable(netif);
		}

		// Set IP configuration for Wifi module
		IPConfiguration conf = new IPConfiguration();
		if (staticIP == null) {
			conf.useDHCP(true);
			if (dns != null) {
				conf.useStaticDNS(true);
				conf.setDNS(new InetAddress[] { InetAddress.getByName(dns) });
			}
		} else {
			conf.useDHCP(false);
			conf.setIP(InetAddress.getByName(staticIP));
			if (netmask != null) {
				conf.setNetmask(InetAddress.getByName(netmask));
			}
			if (gateway != null) {
				conf.setGateway(InetAddress.getByName(gateway));
			}
			if (dns != null) {
				conf.setDNS(new InetAddress[] { InetAddress.getByName(dns) });
			}
		}
		NetworkInterfaceManager.configure(netif, conf);
	}
}