/*******************************************************************************
 * Copyright (c) 2009, 2018 IBM Corp.
 * Copyright 2019-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *    http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Dave Locke - initial API and implementation and/or initial documentation
 *    Ian Craggs - MQTT 3.1.1 support
 *    Ian Craggs - per subscription message handlers (bug 466579)
 *    Ian Craggs - ack control (bug 472172)
 *    MicroEJ Corp. - MicroPaho implementation and optimizations on MicroEJ
 */
package org.eclipse.paho.client.mqttv3;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.SocketTimeoutException;

import javax.net.SocketFactory;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.bon.Util;
import ej.trace.Tracer;

/**
 * Lightweight client for talking to an MQTT server using methods that block until an operation completes.
 *
 * <p>
 * This class implements the blocking {@link IMqttClient} client interface where all actions block until they have
 * completed (or timed out). This implementation is compatible with MicroEJ runtime.
 * </p>
 * <p>
 * An application can connect to an MQTT server using any kind of underlying transport layer that can be created from a
 * {@link SocketFactory} (i.e. bi-directional lossless stream), which is likely one of:
 * </p>
 * <ul>
 * <li>A plain TCP socket
 * <li>A secure SSL/TLS socket
 * </ul>
 *
 * @see IMqttClient
 */
public class MqttClient implements IMqttClient {

	/**
	 * The "MQTT" String constant.
	 */
	protected static final String MQTT = "MQTT"; //$NON-NLS-1$

	private static final String DEBUG_MODE = "micropaho.debug"; //$NON-NLS-1$
	private static final String SERVER_CHECKS = "micropaho.server.check"; //$NON-NLS-1$
	private static final String THREAD_PRIORITY = "micropaho.thread.priority"; //$NON-NLS-1$

	private static final int DEFAULT_PORT_TCP = 1883;
	private static final int DEFAULT_PORT_SSL = 8883;

	private static final String URI_PREFIX_TCP = "tcp"; //$NON-NLS-1$
	private static final String URI_PREFIX_SSL = "ssl"; //$NON-NLS-1$
	private static final String URI_SEP_PREFIX = "://"; //$NON-NLS-1$
	private static final char URI_SEP_PORT = ':';

	private static final int STATE_CONNECTED = 0;
	private static final int STATE_DISCONNECTED = 1;
	private static final int STATE_CLOSED = 2;

	private static final int MSG_CONNECT = 1;
	private static final int MSG_CONNACK = 2;
	private static final int MSG_PUBLISH = 3;
	private static final int MSG_PUBACK = 4;
	private static final int MSG_PUBREC = 5;
	private static final int MSG_PUBREL = 6;
	private static final int MSG_PUBCOMP = 7;
	private static final int MSG_SUBSCRIBE = 8;
	private static final int MSG_SUBACK = 9;
	private static final int MSG_UNSUBSCRIBE = 10;
	private static final int MSG_UNSUBACK = 11;
	private static final int MSG_PINGREQ = 12;
	private static final int MSG_PINGRESP = 13;
	private static final int MSG_DISCONNECT = 14;

	/**
	 * MQTT SUBACK return code - Success - Maximum QoS 0 (§3.9.3).
	 */
	private static final int MQTT_SUBACK_RETURN_CODE_SUCCESS_QOS0 = 0x00;

	/**
	 * MQTT SUBACK return code - Success - Maximum QoS 1 (§3.9.3).
	 */
	private static final int MQTT_SUBACK_RETURN_CODE_SUCCESS_QOS1 = 0x01;

	/**
	 * MQTT SUBACK return code - Failure (§3.9.3).
	 */
	private static final int MQTT_SUBACK_RETURN_CODE_FAILURE = 0x80;

	/**
	 * MQTT QoS 0.
	 */
	protected static final int MQTT_QOS0 = 0;

	/**
	 * MQTT QoS 1.
	 */
	protected static final int MQTT_QOS1 = 1;

	/**
	 * MQTT Version 3.1.1.
	 */
	private static final int MQTT_VERSION_3_1_1 = 4;

	/**
	 * MQTT Packet Identifier Min value (non-zero).
	 */
	private static final int MQTT_PACKET_IDENTIFIER_MIN = 1;

	/**
	 * MQTT Packet Identifier Max value (16 bits).
	 */
	private static final int MQTT_PACKET_IDENTIFIER_MAX = 65535;

	/**
	 * MQTT SUBSCRIBE reserved flags (§2.2.2).
	 */
	private static final int MQTT_SUBSCRIBE_RESERVED_FLAGS = 0x02; // 0010

	private static final int MSG_NONE = 0;
	private static final int PACKET_IDENTIFIER_NONE = 0;

	private static final int VARIABLE_BYTE_INT_MAX = 268435455;

	private static final int TRACE_NB_EVENTS = 3;
	private static final int TRACE_EVENT_MESSAGE_SENT = 0;
	private static final int TRACE_EVENT_MESSAGE_RECEIVED = 1;
	private static final int TRACE_EVENT_KEEP_ALIVE_UPDATED = 2;

	@Nullable
	private MqttCallback callback;

	@Nullable
	private Tracer tracer;

	private final String prefix;
	private final String host;
	private final int port;

	private final String clientId;

	/**
	 * Connection state (lock on this instance)
	 */
	private int state = STATE_DISCONNECTED;

	/**
	 * Connected {@link InputStream}. Never null when {@link #state} is {@link #STATE_CONNECTED}.
	 */
	@Nullable
	private InputStream inputStream;

	/**
	 * Connected {@link OutputStream}. Never null when {@link #state} is {@link #STATE_CONNECTED}. Client lock must be
	 * owned before writing to the stream.
	 */
	@Nullable
	private OutputStream outputStream;

	/**
	 * The CommsReceiver thread. Never null when {@link #state} is {@link #STATE_CONNECTED}.
	 */
	@Nullable
	private Thread receiverThread;

	/**
	 * The pending message action that require an acknowledgment, or {@link #MSG_NONE} if there is no pending message to
	 * acknowledge.
	 */
	private int pendingMessageAction;
	/**
	 * The pending message packet identifier that require an acknowledgment, or {@link #PACKET_IDENTIFIER_NONE}if there
	 * is no pending message to acknowledge.
	 */
	private int pendingMessagePacketIdentifier;

	/**
	 * The pending message acknowledgment exception returned by the server, or <code>null</code> if is there is no
	 * pending message or if the acknowledgment returned without error.
	 */
	@Nullable
	protected MqttException pendingMessageAckException;

	private int nextPacketId;

	/**
	 * Client lock must be owned before reading or writing this field.
	 */
	protected long lastOutboundActivityMillis;

	/**
	 * Create an MqttClient that can be used to communicate with an MQTT server.
	 * <p>
	 * The address of a server can be specified on the constructor.
	 *
	 * <p>
	 * The <code>serverURI</code> parameter is typically used with the the <code>clientId</code> parameter to form a
	 * key. The key is used to store and reference messages while they are being delivered. Hence the serverURI
	 * specified on the constructor must still be specified even if a list of servers is specified on an
	 * MqttConnectOptions object. The serverURI on the constructor must remain the same across restarts of the client
	 * for delivery of messages to be maintained from a given client to a given server or set of servers.
	 *
	 * <p>
	 * The address of the server to connect to is specified as a URI. Two types of connection are supported
	 * <code>tcp://</code> for a TCP connection and <code>ssl://</code> for a TCP connection secured by SSL/TLS. For
	 * example:
	 * </p>
	 * <ul>
	 * <li><code>tcp://localhost:1883</code></li>
	 * <li><code>ssl://localhost:8883</code></li>
	 * </ul>
	 * <p>
	 * If the port is not specified, it will default to 1883 for <code>tcp://</code>" URIs, and 8883 for
	 * <code>ssl://</code> URIs.
	 * </p>
	 *
	 * <p>
	 * A client identifier <code>clientId</code> must be specified and be less that 65535 characters. It must be unique
	 * across all clients connecting to the same server. The clientId is used by the server to store data related to the
	 * client, hence it is important that the clientId remain the same when connecting to a server if durable
	 * subscriptions or reliable messaging are required.
	 * <p>
	 * SSL can be configured by supplying an <code>javax.net.ssl.SSLSocketFactory</code> - applications can use
	 * {@link MqttConnectOptions#setSocketFactory(SocketFactory)} to supply a factory with the appropriate SSL settings.
	 *
	 * @param serverURI
	 *            the address of the server to connect to, specified as a URI.
	 * @param clientId
	 *            a client identifier that is unique on the server being connected to
	 * @throws IllegalArgumentException
	 *             if the URI does not start with "tcp://" or "ssl://" or is invalid
	 * @throws MqttException
	 *             if any other problem was encountered
	 */
	public MqttClient(String serverURI, String clientId) throws MqttException {
		if (Constants.getBoolean(DEBUG_MODE)) {
			this.tracer = new Tracer(MQTT, TRACE_NB_EVENTS);
		}
		this.clientId = clientId;

		// ServerURI basic parsing
		// [ssl|tcp]://host[:port]?
		int schemeSepIndex = serverURI.indexOf(URI_SEP_PREFIX);
		if (schemeSepIndex == -1) {
			throw new IllegalArgumentException(serverURI);
		}

		String scheme = serverURI.substring(0, schemeSepIndex);
		if (!(scheme.equals(URI_PREFIX_TCP) || scheme.equals(URI_PREFIX_SSL))) {
			throw new IllegalArgumentException(serverURI);
		}
		int hostStartIndex = schemeSepIndex + 3;
		int portSepIndex = serverURI.indexOf(URI_SEP_PORT, hostStartIndex);
		String host;
		int port;
		if (portSepIndex == -1) {
			host = serverURI.substring(hostStartIndex);
			if (scheme.equals(URI_PREFIX_TCP)) {
				port = DEFAULT_PORT_TCP;
			} else {
				assert scheme.equals(URI_PREFIX_SSL);
				port = DEFAULT_PORT_SSL;
			}
		} else {
			host = serverURI.substring(hostStartIndex, portSepIndex);
			try {
				int portStartIndex = portSepIndex + 1;
				port = Integer.parseInt(serverURI.substring(portStartIndex));
			} catch (NumberFormatException e) {
				throw new IllegalArgumentException(serverURI);
			}
		}
		this.prefix = host;
		this.host = host;
		this.port = port;
		this.nextPacketId = MQTT_PACKET_IDENTIFIER_MIN;
	}

	/*
	 * @see IMqttClient#connect()
	 */
	@Override
	public void connect() throws MqttException {
		this.connect(new MqttConnectOptions());
	}

	/*
	 * @see IMqttClient#connect(MqttConnectOptions)
	 */
	@Override
	public synchronized void connect(MqttConnectOptions options) throws MqttException {
		int state = this.state;
		if (state == STATE_CONNECTED) {
			throw new MqttException(MqttException.REASON_CODE_CLIENT_CONNECTED);
		} else if (state == STATE_CLOSED) {
			throw new MqttException(MqttException.REASON_CODE_CLIENT_CLOSED);
		}

		SocketFactory factory = options.getSocketFactory();
		if (factory == null) {
			factory = SocketFactory.getDefault();
			assert factory != null;
		}

		SocketAddress sockaddr = new InetSocketAddress(this.host, this.port);
		Socket socket;
		try {
			socket = factory.createSocket();// NOSONAR socket closed when the connection is lost (closed by the server)
											// or when the client is disconnected
			socket.connect(sockaddr, options.getConnectionTimeout() * 1000);
		} catch (IOException e) {
			throw new MqttException(MqttException.REASON_CODE_SERVER_CONNECT_ERROR, e);
		}

		InputStream is;
		OutputStream os;
		try {
			is = socket.getInputStream();
			assert is != null;
			this.inputStream = is;
			os = socket.getOutputStream();
			assert os != null;
			this.outputStream = os;
			writeConnect(os, this.clientId, options);
			readConnack(is, options);
		} catch (MqttException e) {
			closeStreams();
			throw e;
		} catch (EOFException e) {
			closeStreams();
			throw new MqttException(MqttException.REASON_CODE_CONNECTION_LOST, e);
		} catch (IOException e) {
			closeStreams();
			throw new MqttException(e);
		}

		// Start the CommsReceiver thread.
		int keepAliveInMs = options.getKeepAliveInterval() * 1000;

		Thread receiverThread = new Thread(new CommsReceiver(this, socket, is, keepAliveInMs), MQTT);
		receiverThread.setPriority(Constants.getInt(THREAD_PRIORITY));
		try {
			receiverThread.start();
		} catch (Throwable e) {
			// Not enough resource to create the native thread
			closeStreams();
			throw new MqttException(e);
		}
		this.receiverThread = receiverThread;

		// here, the client is CONNECTED:
		// - InputStream and OutputStream are open
		// - CommsReceiver thread is running
		assert this.inputStream != null;
		assert this.outputStream != null;
		assert this.receiverThread != null;
		this.state = STATE_CONNECTED;
	}

	/*
	 * @see IMqttClient#disconnect()
	 */
	@Override
	public void disconnect() throws MqttException {
		assert Thread.currentThread() != this.receiverThread;
		Thread receiverThread;
		synchronized (this) {
			int currentState = this.state;
			if (currentState == STATE_CONNECTED) {
				OutputStream os = this.outputStream;
				assert os != null;
				int info = MSG_DISCONNECT << 4;
				try {
					os.write(info);
					writeLength(os, 0);
					os.flush();
					if (Constants.getBoolean(DEBUG_MODE)) {
						traceMessageSent(info);
					}
					this.lastOutboundActivityMillis = Util.platformTimeMillis();
				} catch (IOException e) {
					// IOException during explicit disconnect
					// silently ignore and finish the internal disconnection
				}
				receiverThread = this.receiverThread;
				internalDisconnect();
			} else if (currentState == STATE_DISCONNECTED) {
				throw new MqttException(MqttException.REASON_CODE_CLIENT_ALREADY_DISCONNECTED);
			} else {
				assert currentState == STATE_CLOSED;
				throw new MqttException(MqttException.REASON_CODE_CLIENT_CLOSED);
			}
		}

		// wait until the CommsReceiver thread is terminated, so that the client resources (heap, stack, thread)
		// are fully reclaimed when this method returns.
		assert receiverThread != null;
		try {
			receiverThread.join();
		} catch (InterruptedException e) {
			throw new AssertionError(e);
		}
	}

	/*
	 * @see IMqttClient#subscribe(String)
	 */
	@Override
	public void subscribe(String topicFilter) throws MqttException {
		this.subscribe(topicFilter, MQTT_QOS1);
	}

	/*
	 * @see IMqttClient#subscribe(String, int)
	 */
	@Override
	public void subscribe(String topicFilter, int qos) throws MqttException {
		subscribeOrUnsubscribe(MSG_SUBSCRIBE, topicFilter, qos);
	}

	/**
	 * Subscribe or Unsubscribe to a topic filter (Common method for footprint considerations - code is almost the
	 * same).
	 *
	 * @param action
	 *            message action ({@link #MSG_SUBSCRIBE} or {@link #MSG_UNSUBSCRIBE})
	 * @param topicFilter
	 *            the topic to subscribe or unsubscribe
	 * @param qos
	 *            requested QoS (only for {{@link #MSG_SUBSCRIBE})
	 * @throws MqttException
	 */
	private synchronized void subscribeOrUnsubscribe(int action, String topicFilter, int qos) throws MqttException {
		checkConnected();
		waitForNoPendingMessage();

		byte[] topicFilterUTF8 = getUTF8(topicFilter);
		int len = 2 + 2 + topicFilterUTF8.length; /* packetid + length + topic + */
		boolean subscribe = action == MSG_SUBSCRIBE;
		if (subscribe) {
			++len; // qos
		}

		OutputStream os = this.outputStream;
		assert os != null; // ensured by checkConnected

		int info = action << 4 | MQTT_SUBSCRIBE_RESERVED_FLAGS;

		try {
			os.write(info);
			/* write remaining length */
			writeLength(os, len);
			writeAndRegisterPacketIdentifier(os, action);
			writeUTF8(os, topicFilterUTF8);
			if (subscribe) {
				os.write(qos);
			}
			os.flush();
			if (Constants.getBoolean(DEBUG_MODE)) {
				traceMessageSent(info);
			}
			this.lastOutboundActivityMillis = Util.platformTimeMillis();
		} catch (IOException e) {
			closeStreams();
			throw new MqttException(e);
		}

		waitForServerAcknowledgment();
	}

	/*
	 * @see IMqttClient#unsubscribe(String)
	 */
	@Override
	public void unsubscribe(String topicFilter) throws MqttException {
		subscribeOrUnsubscribe(MSG_UNSUBSCRIBE, topicFilter, MQTT_QOS0);
	}

	/*
	 * @see IMqttClient#publishBlock(String, byte[], int, boolean)
	 */
	@Override
	public void publish(String topic, byte[] payload, int qos, boolean retained) throws MqttException {
		MqttMessage message = new MqttMessage(payload);
		message.setQos(qos);
		message.setRetained(retained);
		this.publish(topic, message);
	}

	/*
	 * @see IMqttClient#publishBlock(String, MqttMessage)
	 */
	@Override
	public synchronized void publish(String topic, MqttMessage message) throws MqttException {
		checkConnected();
		waitForNoPendingMessage();

		byte[] topicUTF8 = getUTF8(topic);
		int qos = message.getQos();

		byte[] payload = message.getPayload();
		int len = 2 + topicUTF8.length + payload.length;
		if (qos > 0) {
			len += 2; // packet id
		}

		OutputStream os = this.outputStream;
		assert os != null; // ensured by checkConnected

		int info = MSG_PUBLISH << 4 | qos << 1;
		if (message.isRetained()) {
			info |= 0x01;
		}

		try {
			os.write(info);
			writeLength(os, len);
			writeUTF8(os, topicUTF8);
			if (qos > 0) {
				writeAndRegisterPacketIdentifier(os, MSG_PUBLISH);
			}
			os.write(payload);
			os.flush();
			if (Constants.getBoolean(DEBUG_MODE)) {
				traceMessageSent(info);
			}
			this.lastOutboundActivityMillis = Util.platformTimeMillis();
		} catch (IOException e) {
			closeStreams();
			throw new MqttException(e);
		}

		if (qos > 0) {
			waitForServerAcknowledgment();
		}
	}

	private void writeAndRegisterPacketIdentifier(OutputStream os, int message) throws IOException {
		int msgId = getNextPacketId();
		this.pendingMessageAction = message;
		this.pendingMessagePacketIdentifier = msgId;
		writeU16(os, msgId);
	}

	/**
	 * Caller must own the client lock.
	 */
	private void waitForNoPendingMessage() {
		while (this.pendingMessageAction != MSG_NONE) {
			try {
				wait();
			} catch (InterruptedException e) {
				throw new AssertionError(e);
			}
		}

		if (this.state == STATE_DISCONNECTED) {
			throw new MqttException(MqttException.REASON_CODE_CLIENT_NOT_CONNECTED);
		}
	}

	/**
	 * Caller must own the client lock.
	 *
	 * @throws MqttException
	 */
	private void waitForServerAcknowledgment() {
		try {
			wait();
		} catch (InterruptedException e) {
			throw new AssertionError(e);
		}
		// The thread may have been unlocked because
		// - acknowledgment received
		// - the client is disconnected (see disconnect)
		MqttException currentPendingMessageAckException = this.pendingMessageAckException;

		// reset fields
		this.pendingMessageAction = MSG_NONE;
		this.pendingMessagePacketIdentifier = PACKET_IDENTIFIER_NONE;
		this.pendingMessageAckException = null;

		if (this.state == STATE_DISCONNECTED) {
			throw new MqttException(MqttException.REASON_CODE_CLIENT_NOT_CONNECTED);
		} else {
			// unlock at most one thread waiting for sending an other message
			notify();

			if (currentPendingMessageAckException != null) {
				throw currentPendingMessageAckException;
			}
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.paho.client.mqttv3.IMqttClient#close()
	 */
	@Override
	public synchronized void close() throws MqttException {
		int currentState = this.state;
		if (currentState == STATE_CONNECTED) {
			throw new MqttException(MqttException.REASON_CODE_CLIENT_CONNECTED);
		} else if (currentState == STATE_DISCONNECTED) {
			this.state = STATE_CLOSED;
		} else {
			assert currentState == STATE_CLOSED;
			// nop
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.paho.client.mqttv3.IMqttClient#getClientId()
	 */
	@Override
	public String getClientId() {
		return this.clientId;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.paho.client.mqttv3.IMqttClient#getServerURI()
	 */
	@Override
	public String getServerURI() {
		return this.prefix + URI_SEP_PREFIX + this.host + URI_SEP_PORT + this.port;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.paho.client.mqttv3.IMqttClient#isConnected()
	 */
	@Override
	public synchronized boolean isConnected() {
		return this.state == STATE_CONNECTED;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.paho.client.mqttv3.IMqttClient#setCallback(org.eclipse.paho.client.mqttv3.MqttCallback)
	 */
	@Override
	public void setCallback(MqttCallback callback) {
		this.callback = callback;
	}

	/**
	 * Check the client is currently connected. Caller must have taken the lock on this client
	 */
	private void checkConnected() {
		if (this.state != STATE_CONNECTED) {
			throw new MqttException(MqttException.REASON_CODE_CLIENT_NOT_CONNECTED);
		}
	}

	/**
	 * Close and release both {@link #inputStream} and {@link #outputStream}. Do nothing when the stream is null. Errors
	 * on {@link Closeable#close()} are silently ignored.
	 */
	private void closeStreams() {
		InputStream inputStream = this.inputStream;
		if (inputStream != null) {
			try {
				inputStream.close();
			} catch (IOException e) {
				// silently ignore
			}
			this.inputStream = null;
		}

		OutputStream outputStream = this.outputStream;
		if (outputStream != null) {
			try {
				outputStream.close();
			} catch (IOException e) {
				// silently ignore
			}
			this.inputStream = null;
		}
	}

	/**
	 * Close resources, set the client state to {@link #STATE_DISCONNECTED} and unlock all threads waiting for an
	 * acknowledgment. Does nothing if the client is not in the {@link #STATE_CONNECTED} state. Caller must own the
	 * client lock.
	 */
	private void internalDisconnect() {
		if (this.state == STATE_CONNECTED) {
			closeStreams();
			this.receiverThread = null;
			this.state = STATE_DISCONNECTED;
			if (this.pendingMessageAction != MSG_NONE) {
				// unlock all waiting threads
				this.notifyAll();
			}
		} else {
			// disconnect called in parallel by the CommsReceiver thread
			// (IOException received during an explicit disconnect)
			// silently ignore
		}
	}

	/**
	 * Called by the {@link CommsReceiver} thread when an unexpected exception has occured.
	 *
	 * @param reason
	 *            the exception occurred
	 */
	private synchronized void connectionLost(MqttException reason) {
		MqttCallback callback = this.callback;
		if (callback != null) {
			try {
				callback.connectionLost(reason);
			} catch (Throwable e) {
				if (Constants.getBoolean(DEBUG_MODE)) {
					e.printStackTrace();
				}
			}
		}
		internalDisconnect();
	}

	/**
	 * {@link CommsReceiver} is dedicated to read and dispatch incoming messages and to send the <code>PINGREQ</code>
	 * when nothing
	 */
	private class CommsReceiver implements Runnable {

		private final MqttClient client;
		private final Socket socket;
		private final InputStream inputStream;
		private final int keepAliveMillis;
		private boolean pingOutstanding;

		public CommsReceiver(MqttClient client, Socket socket, InputStream inputStream, int keepAliveMillis) {
			this.client = client;
			this.socket = socket;
			this.inputStream = inputStream;
			this.keepAliveMillis = keepAliveMillis;
		}

		@Override
		public void run() {
			InputStream inputStream = this.inputStream;
			boolean keepAliveEnabled = this.keepAliveMillis > 0;
			MqttClient client = this.client;
			try {
				while (true) {
					if (keepAliveEnabled) {
						updateKeepAlive(client);
					}

					// start to read a new message
					int header;
					try {
						header = read(inputStream);
					} catch (SocketTimeoutException e) {
						// read has been unlocked - it is time to send a PINGREQ
						assert keepAliveEnabled;
						continue;
					}
					int packetType = header >>> 4;

					if (Constants.getBoolean(DEBUG_MODE)) {
						Tracer tracer = client.tracer;
						assert tracer != null;
						tracer.recordEvent(TRACE_EVENT_MESSAGE_RECEIVED, packetType);
					}

					int length = readLength(inputStream);
					switch (packetType) {
					case MSG_PUBACK: {
						assert length == 2;
						int packetIdentifier = readU16(inputStream);
						notifyAck(MSG_PUBLISH, packetIdentifier, null);
						break;
					}
					case MSG_SUBACK: {
						// this client only implements subscription of one topic at a time
						assert length == 3;
						int packetIdentifier = readU16(inputStream);
						int returnCode = read(inputStream);
						MqttException ackError;
						if (returnCode == MQTT_SUBACK_RETURN_CODE_FAILURE) {
							ackError = new MqttException(MqttException.REASON_CODE_SUBSCRIBE_FAILED);
						} else {
							if (Constants.getBoolean(SERVER_CHECKS)) {
								if (returnCode != MQTT_SUBACK_RETURN_CODE_SUCCESS_QOS0
										&& returnCode != MQTT_SUBACK_RETURN_CODE_SUCCESS_QOS1) {
									ackError = new MqttException(MqttException.REASON_CODE_SUBSCRIBE_FAILED);
								}
							}
							ackError = null;
						}
						notifyAck(MSG_SUBSCRIBE, packetIdentifier, ackError);
						break;
					}
					case MSG_UNSUBACK: {
						assert length == 2;
						int packetIdentifier = readU16(inputStream);
						notifyAck(MSG_UNSUBSCRIBE, packetIdentifier, null);
						break;
					}
					case MSG_PUBLISH: {
						boolean retained = (header & 0x01) != 0;
						int qos = (header >>> 1) & 0x3;

						byte[] topicUTF8 = readUTF8Bytes(inputStream);
						int currentLength = 2 + topicUTF8.length;
						int packetid = 0;
						if (qos > 0) {
							packetid = readU16(inputStream);
							currentLength += 2;
						}
						int payloadLength = length - currentLength;
						byte[] payload = new byte[payloadLength];
						readFully(inputStream, payload);

						if (qos > 0) {
							// send PUBACK
							synchronized (client) {
								if (client.state == STATE_CONNECTED) {
									OutputStream os = client.outputStream;
									assert os != null;
									int info = MSG_PUBACK << 4;
									os.write(info);
									writeLength(os, 2);
									writeU16(os, packetid);
									os.flush();
									client.lastOutboundActivityMillis = Util.platformTimeMillis();
									if (Constants.getBoolean(DEBUG_MODE)) {
										client.traceMessageSent(info);
									}
								} else {
									// Stream has been closed in parallel - silently ignore
								}
							}
						}

						MqttCallback callback = client.callback;
						if (callback != null) {
							String topic = new String(topicUTF8, "UTF-8"); //$NON-NLS-1$
							MqttMessage message = new MqttMessage(payload);
							message.setQos(qos);
							message.setRetained(retained);
							callback.messageArrived(topic, message);
						}
						break;
					}
					case MSG_PINGRESP: {
						if (Constants.getBoolean(SERVER_CHECKS)) {
							if (!this.pingOutstanding) {
								throw new MqttException(MqttException.REASON_CODE_SERVER_UNEXPECTED_PINGRESP_MESSAGE);
							}
						}
						this.pingOutstanding = false;
						break;
					}
					case MSG_PUBREC:
					case MSG_PUBREL:
					case MSG_PUBCOMP: // QoS 2 packet types not supported
					case MSG_CONNACK: // CONNACK is handled during connection
					default: {
						throw new MqttException(MqttException.REASON_CODE_INVALID_MESSAGE);
					}
					}
				}
			} catch (MqttException e) {
				client.connectionLost(e);
			} catch (Exception e) {
				client.connectionLost(new MqttException(MqttException.REASON_CODE_CONNECTION_LOST, e));
			}
		}

		/**
		 * Update keep alive timeout, send keep alive packet if necessary and prepare the next {@link Socket} read
		 * timeout.
		 */
		private void updateKeepAlive(MqttClient client) throws IOException {
			int timeToNextPingReqMillis;
			synchronized (client) {
				// lock is taken here for the following reasons
				// - access to client.state field
				// - access to client.lastOutboundActivityMillis field
				// - potential send PINGREQ message

				if (client.state != STATE_CONNECTED) {
					// client has been disconnected/closed in parallel - silently ignore
					return;
				}

				// calculate the remaining amount of time before sending a new PINGREQ frame
				long currentTimeMillis = Util.platformTimeMillis();
				int timeSinceLastOutboundActivityMillis = (int) (currentTimeMillis - client.lastOutboundActivityMillis);
				assert timeSinceLastOutboundActivityMillis >= 0;
				timeToNextPingReqMillis = this.keepAliveMillis - timeSinceLastOutboundActivityMillis;
				if (timeToNextPingReqMillis <= 0) {
					if (this.pingOutstanding) {
						// A ping has already been sent. At this point, assume that the
						// broker has hung and the TCP layer hasn't noticed.
						throw new MqttException(MqttException.REASON_CODE_CLIENT_TIMEOUT);
					} else {
						// send a PINGREQ

						OutputStream os = client.outputStream;
						assert os != null;
						int info = MSG_PINGREQ << 4;
						os.write(info);
						/* write remaining length */
						writeLength(os, 0);
						os.flush();
						client.lastOutboundActivityMillis = Util.platformTimeMillis();
						if (Constants.getBoolean(DEBUG_MODE)) {
							client.traceMessageSent(info);
						}
						timeToNextPingReqMillis = this.keepAliveMillis;
						this.pingOutstanding = true;
					}
				}
			}
			assert timeToNextPingReqMillis > 0;
			this.socket.setSoTimeout(timeToNextPingReqMillis);
			if (Constants.getBoolean(DEBUG_MODE)) {
				Tracer tracer = client.tracer;
				assert tracer != null;
				tracer.recordEvent(TRACE_EVENT_KEEP_ALIVE_UPDATED, timeToNextPingReqMillis);
			}
		}

		private void notifyAck(int expectedAction, int packetIdentifier, @Nullable MqttException e) throws IOException {
			MqttClient client = this.client;
			synchronized (client) {
				if (Constants.getBoolean(SERVER_CHECKS)) {
					if (client.pendingMessageAction != expectedAction) {
						throw new MqttException(MqttException.REASON_CODE_SERVER_UNEXPECTED_ACK_MESSAGE);
					}
					if (client.pendingMessagePacketIdentifier != packetIdentifier) {
						throw new MqttException(MqttException.REASON_CODE_SERVER_UNEXPECTED_ACK_PACKET_IDENTIFIER);
					}
				}
				client.pendingMessageAckException = e;
				client.notifyAll();
			}
		}
	}

	/**
	 * Encodes the message length according to the MQTT algorithm
	 *
	 * @param os
	 *            the stream into which the encoded data is written
	 * @param number
	 *            the length to be encoded
	 */
	private static void writeLength(OutputStream os, long number) throws IOException {
		assert (number >= 0 && number <= VARIABLE_BYTE_INT_MAX);
		int numBytes = 0;
		long no = number;
		// Encode the remaining length fields in the four bytes
		do {
			byte digit = (byte) (no % 128);
			no = no / 128;
			if (no > 0) {
				digit |= 0x80;
			}
			os.write(digit);
			numBytes++;
		} while ((no > 0) && (numBytes < 4));

	}

	static final int MAX_NO_OF_REMAINING_LENGTH_BYTES = 4;

	/**
	 * Decodes the message length according to the MQTT algorithm
	 *
	 * @param getcharfn
	 *            pointer to function to read the next character from the data source
	 * @param value
	 *            the decoded length returned
	 * @return the decoded length
	 */
	private static int readLength(InputStream is) throws IOException {
		int digit;
		int msgLength = 0;
		int multiplier = 1;

		do {
			digit = read(is);
			msgLength += ((digit & 0x7F) * multiplier);
			multiplier *= 128;
		} while ((digit & 0x80) != 0);

		if (Constants.getBoolean(SERVER_CHECKS)) {
			if (msgLength < 0 || msgLength > VARIABLE_BYTE_INT_MAX) {
				throw new MqttException(MqttException.REASON_CODE_SERVER_MESSAGE_LENGTH_OVERFLOW);
			}
		}

		return msgLength;
	}

	/**
	 * Reads an unsigned integer from two bytes read from the input buffer
	 *
	 * @param pptr
	 *            pointer to the input buffer - incremented by the number of bytes used & returned
	 * @return the integer value calculated
	 */
	private static int readU16(InputStream is) throws IOException {
		int r1 = read(is);
		int r2 = read(is);
		int len = r1 << 8 | r2;
		return len;
	}

	/**
	 * Read an MQTT String to a byte array
	 */
	private static byte[] readUTF8Bytes(InputStream is) throws IOException {
		int length = readU16(is); /* increments pptr to point past length */
		byte[] bytes = new byte[length];
		readFully(is, bytes);
		return bytes;
	}

	/**
	 * Reads one unsigned byte from the input stream.
	 *
	 * @param is
	 * @return the unsigned byte read
	 * @throws IOException
	 *             if an I/O error occurs
	 * @throws EOFException
	 *             in case the {@link InputStream} is closed or an I/O error occurs
	 */
	private static int read(InputStream is) throws IOException {
		int res = is.read();
		if (res == -1) {
			throw new EOFException();
		}
		return res;
	}

	private static byte[] getUTF8(String str) {
		try {
			return str.getBytes("UTF-8"); //$NON-NLS-1$
		} catch (UnsupportedEncodingException e) {
			// this library require UTF-8 encoding to work properly
			throw new AssertionError(e);
		}
	}

	private static void writeUTF8(OutputStream os, byte[] encodedString) throws IOException {
		writeU16(os, encodedString.length);
		os.write(encodedString);
	}

	/**
	 * @param length
	 */
	private static void writeU16(OutputStream os, int value) throws IOException {
		assert (value & 0xFFFF) == value;
		// OutputStream.write spec: The 24 high-order bits of b are ignored.
		os.write(value >>> 8);
		os.write(value);
	}

	/**
	 * Serializes the connect options into the buffer. Caller must own the client lock.
	 *
	 * @param buf
	 *            the buffer into which the packet will be serialized
	 * @param len
	 *            the length in bytes of the supplied buffer
	 * @param options
	 *            the options to be used to build the connect packet
	 */
	private void writeConnect(OutputStream os, String clientId, MqttConnectOptions options) throws IOException {
		// compute the expected packet length according to given options
		int len = 10; // MQTT Version 4 (MQTT_VERSION_3_1_1)
		byte[] clientIdUTF8 = getUTF8(clientId);

		len += clientIdUTF8.length + 2;
		String userName = options.getUserName();
		byte[] userNameUTF8 = null;
		byte[] passwordUTF8 = null;
		if (userName != null) {
			userNameUTF8 = getUTF8(userName);
			len += userNameUTF8.length + 2;
			char[] password = options.getPassword();
			if (password != null) {
				passwordUTF8 = getUTF8(new String(password));
				len += passwordUTF8.length + 2;
			}
		}
		int info = MSG_CONNECT << 4;
		os.write(info);

		/* write remaining length */
		writeLength(os, len);

		writeUTF8(os, MQTT.getBytes());
		os.write(MQTT_VERSION_3_1_1);

		int flags = 0;

		if (options.isCleanSession()) {
			flags |= 0x02;
		}

		if (userNameUTF8 != null) {
			flags |= 0x80;
			if (passwordUTF8 != null) {
				flags |= 0x40;
			}
		}

		os.write(flags);
		writeU16(os, options.getKeepAliveInterval());
		writeUTF8(os, clientIdUTF8);

		if (userNameUTF8 != null) {
			writeUTF8(os, userNameUTF8);
			if (passwordUTF8 != null) {
				writeUTF8(os, passwordUTF8);
			}
		}
		os.flush();
		this.lastOutboundActivityMillis = Util.platformTimeMillis();
		if (Constants.getBoolean(DEBUG_MODE)) {
			traceMessageSent(info);
		}
	}

	/**
	 * Read the expected {{@link #MSG_CONNACK} packet.<br>
	 * As it is the first received packet, all bytes received from the server are checked in order to fail if the
	 * connected server does not speak MQTT protocol.
	 */
	private void readConnack(InputStream is, MqttConnectOptions options) throws IOException {
		int header = read(is);
		int packetType = header >>> 4;

		if (packetType < MSG_CONNECT || packetType > MSG_DISCONNECT) {
			throw new MqttException(MqttException.REASON_CODE_INVALID_MESSAGE);
		}

		if (Constants.getBoolean(DEBUG_MODE)) {
			Tracer tracer = this.tracer;
			assert tracer != null;
			tracer.recordEvent(TRACE_EVENT_MESSAGE_RECEIVED, packetType);
		}

		if (packetType != MSG_CONNACK) {
			// [MQTT-3.2.0-1]
			throw new MqttException(MqttException.REASON_CODE_INVALID_MESSAGE);
		}

		int length = readLength(is);
		if (length != 2) {
			throw new MqttException(MqttException.REASON_CODE_INVALID_MESSAGE);
		}

		boolean sessionPresent = (read(is) & 0x01) == 0x01;
		if (Constants.getBoolean(SERVER_CHECKS)) {
			if (options.isCleanSession() && sessionPresent) {
				throw new MqttException(MqttException.REASON_CODE_SERVER_EXPECTED_CLEAN_SESSION);
			}
		}
		int returnCode = read(is);
		if (returnCode != 0) {
			throw new MqttException(returnCode);
		}
	}

	private int getNextPacketId() {
		return this.nextPacketId = (this.nextPacketId == MQTT_PACKET_IDENTIFIER_MAX) ? 1 : this.nextPacketId + 1;
	}

	private void traceMessageSent(int info) {
		assert Constants.getBoolean(DEBUG_MODE);
		Tracer tracer = this.tracer;
		assert tracer != null;
		long time = this.lastOutboundActivityMillis;
		tracer.recordEvent(TRACE_EVENT_MESSAGE_SENT, (int) (time >>> 32), (int) time, info >>> 4);
	}

	private static void readFully(InputStream is, byte[] array) throws IOException {
		int length = array.length;
		int ptr = 0;
		while (ptr < length) {
			int nb = is.read(array, ptr, length - ptr);
			if (nb == -1) {
				throw new EOFException();
			}
			ptr += nb;
		}
	}
}
