/*
 * Copyright (c) 2001, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2019 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package sun.net.www.protocol.https;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.net.Socket;
import java.net.SocketException;
import java.net.URL;
import java.net.UnknownHostException;

import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

import sun.net.www.http.HttpClient;
import sun.net.www.protocol.http.HttpURLConnection;


/**
 * This class provides HTTPS client URL support, building on the standard
 * "sun.net.www" HTTP protocol handler.  HTTPS is the same protocol as HTTP,
 * but differs in the transport layer which it uses:  <UL>
 *
 *      <LI>There's a <em>Secure Sockets Layer</em> between TCP
 *      and the HTTP protocol code.
 *
 *      <LI>It uses a different default TCP port.
 *
 *      <LI>It doesn't use application level proxies, which can see and
 *      manipulate HTTP user level data, compromising privacy.  It uses
 *      low level tunneling instead, which hides HTTP protocol and data
 *      from all third parties.  (Traffic analysis is still possible).
 *
 *      <LI>It does basic server authentication, to protect
 *      against "URL spoofing" attacks.  This involves deciding
 *      whether the X.509 certificate chain identifying the server
 *      is trusted, and verifying that the name of the server is
 *      found in the certificate.  (The application may enable an
 *      anonymous SSL cipher suite, and such checks are not done
 *      for anonymous ciphers.)
 *
 *      <LI>It exposes key SSL session attributes, specifically the
 *      cipher suite in use and the server's X509 certificates, to
 *      application software which knows about this protocol handler.
 *
 *      </UL>
 *
 * <P> System properties used include:  <UL>
 *
 *      <LI><em>https.proxyHost</em> ... the host supporting SSL
 *      tunneling using the conventional CONNECT syntax
 *
 *      <LI><em>https.proxyPort</em> ... port to use on proxyHost
 *
 *      <LI><em>https.cipherSuites</em> ... comma separated list of
 *      SSL cipher suite names to enable.
 *
 *      <LI><em>http.nonProxyHosts</em> ...
 *
 *      </UL>
 *
 * @author David Brownell
 * @author Bill Foote
 */

// final for export control reasons (access to APIs); remove with care
final class HttpsClient extends HttpClient
{
	// STATIC STATE and ACCESSORS THERETO

	// HTTPS uses a different default port number than HTTP.
	private static final int    httpsPortNumber = 443;


	/** Returns the default HTTPS port (443) */
	@Override
	protected int getDefaultPort() { return httpsPortNumber; }

	private SSLSocketFactory sslSocketFactory;

	// HttpClient.proxyDisabled will always be false, because we don't
	// use an application-level HTTP proxy.  We might tunnel through
	// our http proxy, though.


	// INSTANCE DATA


	// CONSTRUCTOR, FACTORY


	HttpsClient(SSLSocketFactory sf, URL url,
			int connectTimeout)
					throws IOException {
		setSSLSocketFactory(sf);

		this.host = url.getHost();
		this.url = url;
		port = url.getPort();
		if (port == -1) {
			port = getDefaultPort();
		}
		setConnectTimeout(connectTimeout);
		openServer();
	}



	/** See HttpClient for the model for this method. */
	static HttpClient New(SSLSocketFactory sf, URL url,
			HttpURLConnection httpuc) throws IOException {
		return HttpsClient.New(sf, url, -1, httpuc);
	}

	/**
	 * Get a HTTPS client to the URL.
	 */

	static HttpClient New(SSLSocketFactory sf, URL url,
			int connectTimeout, HttpURLConnection httpuc)
					throws IOException
	{
		HttpsClient ret = new HttpsClient(sf, url, connectTimeout);

		return ret;
	}

	// METHODS

	void setSSLSocketFactory(SSLSocketFactory sf) {
		sslSocketFactory = sf;
	}

	SSLSocketFactory getSSLSocketFactory() {
		return sslSocketFactory;
	}

	/**
	 * The following method, createSocket, is defined in NetworkClient
	 * and overridden here so that the socket facroty is used to create
	 * new sockets.
	 */
	@Override
	protected Socket createSocket() throws IOException {
		try {
			return sslSocketFactory.createSocket();
		} catch (SocketException se) {
			//
			// bug 6771432
			// javax.net.SocketFactory throws a SocketException with an
			// UnsupportedOperationException as its cause to indicate that
			// unconnected sockets have not been implemented.
			//
			Throwable t = se.getCause();
			if (t != null && t instanceof UnsupportedOperationException) {
				return super.createSocket();
			} else {
				throw se;
			}
		}
	}



	//@Override
	public void afterConnect() throws IOException, UnknownHostException {
		SSLSocket s = null;
		SSLSocketFactory factory = sslSocketFactory;
		try {
			// Recreate a SSLSocket even if serverSocket already is one,
			// to be sure that we have the info about the host (TLS SNI cannot work
			// otherwise)
			s = (SSLSocket) factory.createSocket(serverSocket,
					host, port, true);
		} catch (IOException ex) {
			// If we fail to connect through the tunnel, try it
			// locally, as a last resort.  If this doesn't work,
			// throw the original exception.
			try {
				s = (SSLSocket)factory.createSocket(host, port);
			} catch (IOException ignored) {
				throw ex;
			}
		}

		//
		// Force handshaking, so that we get any authentication.
		// Register a handshake callback so our session state tracks any
		// later session renegotiations.
		//

		// We have two hostname verification approaches. One is in
		// SSL/TLS socket layer, where the algorithm is configured with
		// SSLParameters.setEndpointIdentificationAlgorithm(), and the
		// hostname verification is done by X509ExtendedTrustManager when
		// the algorithm is "HTTPS". The other one is in HTTPS layer,
		// where the algorithm is customized by
		// HttpsURLConnection.setHostnameVerifier(), and the hostname
		// verification is done by HostnameVerifier when the default
		// rules for hostname verification fail.
		//
		// The relationship between two hostname verification approaches
		// likes the following:
		//
		//               |             EIA algorithm
		//               +----------------------------------------------
		//               |     null      |   HTTPS    |   LDAP/other   |
		// -------------------------------------------------------------
		//     |         |1              |2           |3               |
		// HNV | default | Set HTTPS EIA | use EIA    | HTTPS          |
		//     |--------------------------------------------------------
		//     | non -   |4              |5           |6               |
		//     | default | HTTPS/HNV     | use EIA    | HTTPS/HNV      |
		// -------------------------------------------------------------
		//
		// Abbreviation:
		//     EIA: the endpoint identification algorithm in SSL/TLS
		//           socket layer
		//     HNV: the hostname verification object in HTTPS layer
		// Notes:
		//     case 1. default HNV and EIA is null
		//           Set EIA as HTTPS, hostname check done in SSL/TLS
		//           layer.
		//     case 2. default HNV and EIA is HTTPS
		//           Use existing EIA, hostname check done in SSL/TLS
		//           layer.
		//     case 3. default HNV and EIA is other than HTTPS
		//           Use existing EIA, EIA check done in SSL/TLS
		//           layer, then do HTTPS check in HTTPS layer.
		//     case 4. non-default HNV and EIA is null
		//           No EIA, no EIA check done in SSL/TLS layer, then do
		//           HTTPS check in HTTPS layer using HNV as override.
		//     case 5. non-default HNV and EIA is HTTPS
		//           Use existing EIA, hostname check done in SSL/TLS
		//           layer. No HNV override possible. We will review this
		//           decision and may update the architecture for JDK 7.
		//     case 6. non-default HNV and EIA is other than HTTPS
		//           Use existing EIA, EIA check done in SSL/TLS layer,
		//           then do HTTPS check in HTTPS layer as override.

		s.startHandshake();
		// change the serverSocket and serverOutput
		serverSocket = s;
		try {
			serverOutput = new PrintStream(
					new BufferedOutputStream(serverSocket.getOutputStream()),
					false, encoding);
		} catch (UnsupportedEncodingException e) {
			throw new InternalError(encoding+" encoding not found");
		}
	}

}
