/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector.combinator;

import ej.mwt.Container;
import ej.mwt.Widget;
import ej.mwt.stylesheet.selector.Selector;

/**
 * An adjacent sibling combinator selects by checking the widget immediate sibling.
 * <p>
 * Equivalent to <code>widget+widget</code> selector in CSS.
 *
 * @see Widget#getParent()
 */
public class AdjacentSiblingCombinator extends Combinator {

	/**
	 * Creates an adjacent sibling combinator.
	 *
	 * @param selector1
	 *            the first selector.
	 * @param selector2
	 *            the second selector.
	 */
	public AdjacentSiblingCombinator(Selector selector1, Selector selector2) {
		super(selector1, selector2);
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		// check that second selector applies to widget
		if (!getSecondSelector().appliesToWidget(widget)) {
			return false;
		}

		// get parent
		Container parent = widget.getParent();
		if (parent == null) {
			return false;
		}

		// get widget index in parent
		int index = parent.getChildIndex(widget);
		if (index == 0) {
			return false;
		}

		// check that first selector applies to previous sibling widget
		Widget siblingWidget = parent.getChild(index - 1);
		return getFirstSelector().appliesToWidget(siblingWidget);
	}
}
