/*
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import ej.annotation.Nullable;
import ej.basictool.ArrayTools;
import ej.basictool.BitFieldHelper;
import ej.basictool.ThreadUtils;
import ej.microui.MicroUI;
import ej.microui.display.GraphicsContext;
import ej.mwt.style.Style;
import ej.mwt.style.dimension.Dimension;
import ej.mwt.util.OutlineHelper;
import ej.mwt.util.Rectangle;
import ej.mwt.util.Size;

/**
 * Widget is the superclass of all the user interface objects.
 * <p>
 * There are a number of important concepts involving widgets:
 * <ul>
 * <li>Lay out.
 * <p>
 * Lay out is the process of positioning and setting the size of the widgets on a desktop.
 * <p>
 * This is performed by evaluating the optimal size of each widget and container in the hierarchy then, considering the
 * layout of the containers and the optimal size of their children, setting the position and actual size of all widgets.
 * <p>
 * The optimal size is the minimal size that allow to show correctly the content of a widget. For instance the optimal
 * size of a label that displays a string will be the size of the string with the font defined in the style.
 * <p>
 * The optimal size is computed by {@link #computeContentOptimalSize(Size)} method. On this size are applied the boxes
 * and dimension defined in the style, and the size of the widget is set. This size is then used by the parent container
 * to lay out (set position and size) the widget along with its siblings.
 * <p>
 * Whenever the state of a widget changes in a way that may affect its optimal size and the lay out of the desktop of
 * which it is a part then the hierarchy of the widget must be ask to perform a new lay out. This can be achieved by
 * invoking {@link #requestLayOut()} on one of the parents of the widget or the desktop depending on the lay out
 * modification.
 * <p>
 * An application will normally invoke {@link #requestLayOut()} after making a set of changes to widgets.
 * <li>Rendering.
 * <p>
 * Any widget can be asked to render itself by invoking {@link #requestRender()}. If a widget has children it will ask
 * them to render. If the widget is transparent it will cause the relevant area of its parent to be rendered. Note that
 * a render request does not trigger a new lay out, and the scope of the rendering that results from a call to
 * {@link #requestRender()} will never exceed the widget itself, its children (recursively), and, if it is transparent,
 * its parent (recursively if the parent is also transparent).</li>
 * </ul>
 * <p>
 * For heap optimization the position and size are stored as a <code>short</code> and therefore are limited between
 * <code>-32768</code> and <code>32767</code>.
 */
public abstract class Widget {

	/**
	 * A width or height hint equal to <code>NO_CONSTRAINT</code> means that there is no constraint on this dimension.
	 *
	 * @see Widget#computeContentOptimalSize(Size)
	 */
	public static final int NO_CONSTRAINT = 0;

	private static final Rectangle SHARED_RECTANGLE = new Rectangle(0, 0, 0, 0);
	private static final int[] EMPTY_INT_ARRAY = new int[0];

	// Widget characteristics flags.
	private static final int ENABLED_SHIFT = 0;
	private static final int ENABLED_MASK = 0x1 << ENABLED_SHIFT;
	private static final int ATTACHED_SHIFT = 1;
	private static final int ATTACHED_MASK = 0x1 << ATTACHED_SHIFT;
	private static final int SHOWN_SHIFT = 2;
	private static final int SHOWN_MASK = 0x1 << SHOWN_SHIFT;

	// Widget position and size.
	/* package */short x;
	/* package */short y;
	/* package */short width;
	/* package */short height;

	// Widget parent: either a Container or a Desktop.
	@Nullable
	/* package */Object parent;

	// Widget characteristics.
	private byte flags;

	/**
	 * The cached style.
	 */
	@Nullable
	/* package */ Style style;
	/**
	 * The class selectors.
	 */
	private int[] classSelectors;

	/**
	 * Creates a widget.
	 * <p>
	 * Once created, the widget is disabled. It may be enabled by calling {@link #setEnabled(boolean)}. Enabled widgets
	 * can handle events by overriding {@link #handleEvent(int)}.
	 */
	public Widget() {
		// Disabled, opaque, detached and shown by default.
		this.flags = 0;
		this.classSelectors = EMPTY_INT_ARRAY;
	}

	/**
	 * Gets the x coordinate of this widget, relative to its parent.
	 *
	 * @return the x coordinate.
	 */
	public int getX() {
		return this.x;
	}

	/**
	 * Gets the y coordinate of this widget, relative to its parent.
	 *
	 * @return the y coordinate.
	 */
	public int getY() {
		return this.y;
	}

	/**
	 * Gets the width of this widget.
	 *
	 * @return the width.
	 */
	public int getWidth() {
		return this.width;
	}

	/**
	 * Gets the height of this widget.
	 *
	 * @return the height.
	 */
	public int getHeight() {
		return this.height;
	}

	/**
	 * Sets the position of this widget.
	 *
	 * @param x
	 *            the x coordinate to set.
	 * @param y
	 *            the y coordinate to set.
	 */
	public void setPosition(int x, int y) {
		this.x = (short) x;
		this.y = (short) y;
	}

	/**
	 * Sets the size of this widget.
	 *
	 * @param width
	 *            the width to set.
	 * @param height
	 *            the height to set.
	 */
	private void setSize(int width, int height) {
		this.width = (short) Math.max(0, width);
		this.height = (short) Math.max(0, height);
	}

	/**
	 * Gets the content bounds of this widget (the bounds minus the outlines).
	 *
	 * @return the content bounds of this widget.
	 */
	public Rectangle getContentBounds() {
		Style style = this.style;
		assert (style != null);

		Rectangle contentBounds = new Rectangle(0, 0, this.width, this.height);
		OutlineHelper.applyOutlines(contentBounds, style);
		return contentBounds;
	}

	/**
	 * Lays out this widget.
	 * <p>
	 * The style of this widget must be set before (not <code>null</code>).
	 * <p>
	 * The dimension in the style is applied to modify the given bounds before applying to the actual bounds.
	 *
	 * @param x
	 *            the x coordinate.
	 * @param y
	 *            the y coordinate.
	 * @param width
	 *            the width.
	 * @param height
	 *            the height.
	 */
	/* package */ void layOut(int x, int y, int width, int height) {
		Style style = this.style;
		assert style != null;

		// delegate layout to dimension
		Dimension dimension = style.getDimension();
		Rectangle bounds = getSharedRectangle(x, y, width, height);
		dimension.layOut(this, bounds);
		setPosition(bounds.getX(), bounds.getY());
		setSize(bounds.getWidth(), bounds.getHeight());
		onLaidOut();
	}

	/**
	 * This method is called as soon as the widget bounds are set.
	 */
	protected void onLaidOut() {
		// Do nothing by default.
	}

	/**
	 * This method is called as soon as:
	 * <ul>
	 * <li>the widget is attached to a desktop that is attached,</li>
	 * <li>the desktop of the widget is attached.</li>
	 * </ul>
	 * <p>
	 * After this call, the widget is ready to be rendered.
	 * <p>
	 * For example, the widget can allocate some resources useful to render it.
	 *
	 * @see Desktop#setAttached()
	 */
	protected void onAttached() {
		// Do nothing by default.
	}

	/**
	 * This method is called as soon as:
	 * <ul>
	 * <li>the widget is detached from a desktop that is attached,</li>
	 * <li>the desktop of the widget is detached.</li>
	 * </ul>
	 * <p>
	 * After this call, the resources allocated to render the widget must be disposed.
	 *
	 * @see Desktop#setDetached()
	 */
	protected void onDetached() {
		// Do nothing by default.
	}

	/**
	 * Sets the given container or desktop as parent of this widget.
	 *
	 * @throws IllegalArgumentException
	 *             if this widget already has a parent.
	 */
	/* package */void setParent(Object parent, boolean attach) {
		if (this.parent != null) {
			throw new IllegalArgumentException();
		}

		this.parent = parent;

		if (attach) {
			setAttached();
		}
	}

	/**
	 * Resets widget's parent.
	 */
	/* package */void resetParent() {
		setDetached();
		this.parent = null;
	}

	/**
	 * Gets whether this widget is attached or not.
	 * <p>
	 * A widget is considered as attached if it belongs to the hierarchy of an attached desktop.
	 *
	 * @return <code>true</code> if this widget is attached, <code>false</code> otherwise.
	 */
	public boolean isAttached() {
		return BitFieldHelper.getBooleanProperty(this.flags, ATTACHED_MASK);
	}

	/**
	 * @throws IllegalArgumentException
	 *             if the widget is being attached whereas it is already attached.
	 */
	/* package */void setAttached() {
		if (isAttached()) {
			throw new IllegalArgumentException();
		} else {
			this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, ATTACHED_MASK);
			try {
				onAttached();
			} catch (Exception e) {
				ThreadUtils.handleUncaughtException(e);
			}
		}
	}

	/* package */void setDetached() {
		if (isAttached()) {
			setHidden();
			this.flags = (byte) BitFieldHelper.unsetBooleanProperty(this.flags, ATTACHED_MASK);
			try {
				onDetached();
			} catch (Exception e) {
				ThreadUtils.handleUncaughtException(e);
			}
		}
	}

	/**
	 * This method is called as soon as the widget is visible on the display.
	 * <p>
	 * For example, this method can be used to start a task that refreshes the widget periodically.
	 */
	protected void onShown() {
		// Do nothing by default.
	}

	/**
	 * This method is called as soon as the widget is no more visible on the display.
	 * <p>
	 * After this call, all that has been allocated or started in {@link #onShown()} must be disposed or stopped.
	 */
	protected void onHidden() {
		// Do nothing by default.
	}

	/**
	 * Gets whether this widget is shown or not.
	 * <p>
	 * This information is set by the parent of the widget and used to know if the widget can be drawn.
	 *
	 * @return <code>true</code> if this widget is shown, <code>false</code> otherwise.
	 * @see Container#setShownChildren()
	 */
	public boolean isShown() {
		return BitFieldHelper.getBooleanProperty(this.flags, SHOWN_MASK);
	}

	/* package */ void setShown() {
		if (!isShown()) {
			this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, SHOWN_MASK);
			try {
				onShown();
			} catch (Exception e) {
				ThreadUtils.handleUncaughtException(e);
			}
		}
	}

	/* package */ void setHidden() {
		if (isShown()) {
			this.flags = (byte) BitFieldHelper.unsetBooleanProperty(this.flags, SHOWN_MASK);
			try {
				onHidden();
			} catch (Exception e) {
				ThreadUtils.handleUncaughtException(e);
			}
		}
	}

	/**
	 * Gets the absolute x coordinate of the widget. That is, the x coordinate relative to the origin of the display.
	 *
	 * @return the absolute x coordinate of the widget.
	 */
	public int getAbsoluteX() {
		int absoluteX = this.x;
		Container parent = getParent();
		while (parent != null) {
			absoluteX += parent.x + parent.contentX;
			parent = parent.getParent();
		}
		return absoluteX;
	}

	/**
	 * Gets the absolute y coordinate of the widget. That is, the y coordinate relative to the origin of the display.
	 *
	 * @return the absolute y coordinate of the widget.
	 */
	public int getAbsoluteY() {
		int absoluteY = this.y;
		Container parent = getParent();
		while (parent != null) {
			absoluteY += parent.y + parent.contentY;
			parent = parent.getParent();
		}
		return absoluteY;
	}

	/**
	 * Tells whether or not this widget is transparent.
	 * <p>
	 * By default, a widget is transparent. A widget is considered as transparent if it does not draw every pixel of its
	 * bounds with maximal opacity when it is rendered. If a widget is transparent, its parent (recursively if also
	 * transparent) has to be rendered before the widget.
	 *
	 * @return <code>true</code> if this widget is transparent, <code>false</code> otherwise.
	 * @see #contains(int, int)
	 */
	public boolean isTransparent() {
		Style style = this.style;
		assert (style != null);

		Rectangle bounds = getSharedRectangle(this.x, this.y, this.width, this.height);
		style.getMargin().apply(bounds);
		return style.getBackground().isTransparent(bounds.getWidth(), bounds.getHeight());
	}

	/**
	 * Returns whether or not this widget contains the given widget in its hierarchy.
	 * <p>
	 * A widget contains an other widget if one of the children of the former contains the latter or if they reference
	 * the same widget.
	 *
	 * @param widget
	 *            the widget to check.
	 * @return <code>true</code> if this widget contains the given widget, <code>false</code> otherwise.
	 */
	public boolean containsWidget(Widget widget) {
		Widget ancestor = widget;
		do {
			if (ancestor == this) {
				return true;
			}
			ancestor = ancestor.getParent();
		} while (ancestor != null);
		return false;
	}

	/**
	 * Gets whether or not a position (x,y) is in the widget's bounds.
	 * <p>
	 * The given position is considered here as a relative position to parent.
	 * <p>
	 * Subclasses can override this method if the widget is not reactive to pointer events in its entire bounds. As long
	 * as this method is used to dispatch pointer events, its implementation should be as fast as possible (for example
	 * by simplifying the shape of the sensitive area).
	 * <p>
	 * If this method is overridden, it may be relevant to override the {@link #isTransparent()} method as well.
	 *
	 * @param x
	 *            x coordinate.
	 * @param y
	 *            y coordinate.
	 * @return <code>true</code> if the <code>(x,y)</code> position is in widget bounds, <code>false</code> otherwise.
	 */
	public boolean contains(int x, int y) {
		Rectangle bounds = getSharedRectangle(this.x, this.y, this.width, this.height);
		Style style = this.style;
		if (style != null) {
			style.getMargin().apply(bounds);
		}
		int boundsX = bounds.getX();
		int boundsY = bounds.getY();
		return x >= boundsX && x <= boundsX + bounds.getWidth() && y >= boundsY && y <= boundsY + bounds.getHeight();
	}

	/**
	 * Gets the widget at the specified position.
	 * <p>
	 * If this widget does not <code>contains(x, y)</code>, <code>null</code> is returned, else this widget is returned.
	 * The position is considered here as a relative position to parent.
	 *
	 * @param x
	 *            x coordinate.
	 * @param y
	 *            y coordinate.
	 * @return this widget if it <code>contains(x, y)</code>, <code>null</code> otherwise.
	 */
	@Nullable
	public Widget getWidgetAt(int x, int y) {
		if (contains(x, y)) {
			return this;
		}
		return null;
	}

	/**
	 * Gets whether or not this widget is enabled.
	 *
	 * @return <code>true</code> if this widget is enabled, <code>false</code> otherwise.
	 */
	public boolean isEnabled() {
		return BitFieldHelper.getBooleanProperty(this.flags, ENABLED_MASK);
	}

	/**
	 * Sets this widget to be enabled or not.
	 *
	 * @param enabled
	 *            <code>true</code> if this widget is to be enabled, <code>false</code> otherwise.
	 */
	public void setEnabled(boolean enabled) {
		this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, enabled, ENABLED_MASK);
	}

	/**
	 * Requests a lay out of all the widgets in the sub hierarchy of this widget.
	 * <p>
	 * This method returns immediately and the layout of the widget is performed asynchronously in the MicroUI thread.
	 * <p>
	 * This method can only be called if this widget has been added to a desktop.
	 * <p>
	 * Nothing is done if the widget is not attached.
	 * <p>
	 * If the widget was not already shown, it is shown as soon as its bounds are set.
	 * <p>
	 * The style of all the widgets in the hierarchy is set (or updated) during this phase.
	 */
	public void requestLayOut() {
		if (!isAttached()) {
			return;
		}
		MicroUI.callSerially(new Runnable() {
			@Override
			public void run() {
				if (!isAttached()) {
					return;
				}
				Widget widget = Widget.this;
				int width = widget.width;
				int height = widget.height;
				computeOptimalSize(width, height);
				layOut(widget.x, widget.y, width, height);
				setShown();
			}
		});
		// inline requestRender() without the isShown() check
		getDesktop().requestRender(this, 0, 0, this.width, this.height);
	}

	/**
	 * Computes the optimal size of this widget including the outlines defined in the style.
	 * <p>
	 * After this call the style is set and the optimal size will have been established.
	 * <p>
	 * The parameters define the maximum size available for this widget, or {@link Widget#NO_CONSTRAINT} if there is no
	 * constraint.
	 *
	 * @param availableWidth
	 *            the width available for this widget or {@link Widget#NO_CONSTRAINT}.
	 * @param availableHeight
	 *            the height available for this widget or {@link Widget#NO_CONSTRAINT}.
	 */
	/* package */void computeOptimalSize(int availableWidth, int availableHeight) {
		Desktop desktop = getDesktop();
		Style style = desktop.getStylesheet().getStyle(this);
		setStyle(style);

		Dimension dimension = style.getDimension();
		Size size = new Size(availableWidth, availableHeight);
		OutlineHelper.applyOutlines(size, style);

		int horizontalOutlines = availableWidth - size.getWidth();
		int verticalOutlines = availableHeight - size.getHeight();

		if (availableWidth == Widget.NO_CONSTRAINT) {
			size.setWidth(Widget.NO_CONSTRAINT);
		}
		if (availableHeight == Widget.NO_CONSTRAINT) {
			size.setHeight(Widget.NO_CONSTRAINT);
		}

		dimension.getAvailableSize(this, availableWidth, availableHeight, size);
		try {
			computeContentOptimalSize(size);
		} catch (Exception e) {
			ThreadUtils.handleUncaughtException(e);
		}
		dimension.computeOptimalSize(this, availableWidth, availableHeight, size);

		setSize(size.getWidth() + horizontalOutlines, size.getHeight() + verticalOutlines);
	}

	/**
	 * Computes the optimal size of the widget.
	 * <p>
	 * This method does not consider the border, margin, padding and dimension specified in the style.
	 * <p>
	 * The given size is the available size for this widget in its parent. A width or a height equal to
	 * <code>Widget#NO_CONSTRAINT</code> means that there is no constraint on this dimension.
	 * <p>
	 * The given size is modified to set the optimal size.
	 *
	 * @param size
	 *            the size available for the content.
	 */
	protected abstract void computeContentOptimalSize(Size size);

	/**
	 * Requests a render of this entire widget on the display. This method returns immediately and the rendering of the
	 * widget is performed asynchronously in the MicroUI thread.
	 * <p>
	 * If the widget is not shown, nothing is done.
	 *
	 * @see ej.mwt.render.RenderPolicy
	 */
	public void requestRender() {
		requestRender(0, 0, this.width, this.height);
	}

	/**
	 * Requests a render of a zone of this widget on the display. This method returns immediately and the rendering of
	 * the widget is performed asynchronously in the MicroUI thread.
	 * <p>
	 * If the widget is not shown, nothing is done.
	 * <p>
	 * If the given area exceeds the bounds of the widget, only the intersection of the widget and the area will be
	 * rendered.
	 *
	 * @param x
	 *            the relative x coordinate of the area to render.
	 * @param y
	 *            the relative y coordinate of the area to render.
	 * @param width
	 *            the width of the area to render.
	 * @param height
	 *            the height of the area to render.
	 * @see ej.mwt.render.RenderPolicy
	 */
	public void requestRender(int x, int y, int width, int height) {
		if (isShown()) {
			getDesktop().requestRender(this, x, y, width, height);
		}
	}

	/**
	 * Handles the given event. Does nothing by default and returns <code>false</code> (does not consume event).
	 * <p>
	 * Called by the desktop event manager.
	 *
	 * @param event
	 *            the event to handle.
	 * @return <code>true</code> if the widget has consumed the event, <code>false</code> otherwise.
	 */
	public boolean handleEvent(int event) {
		return false;
	}

	/* package */ void paint(GraphicsContext g) {
		if (!isAttached()) {
			return;
		}
		g.translate(this.x, this.y);
		g.intersectClip(0, 0, this.width, this.height);
		if (g.getClipWidth() == 0 || g.getClipHeight() == 0) {
			return; // Nothing to paint.
		}
		render(g);
	}

	/**
	 * Renders the widget on the given graphics context.
	 * <p>
	 * The given graphics context is translated to the origin of the widget and clipped to the area to draw.
	 * <p>
	 * First, the different outlines defined in the style are applied, then, the content is rendered.
	 *
	 * @param g
	 *            the graphics context to use to draw the widget.
	 * @see OutlineHelper#applyOutlinesAndBackground(GraphicsContext, Size, Style)
	 * @see #renderContent(GraphicsContext, int, int)
	 */
	public void render(GraphicsContext g) {
		Style style = this.style;
		assert style != null;
		Size contentSize = new Size(this.width, this.height);
		OutlineHelper.applyOutlinesAndBackground(g, contentSize, style);
		try {
			renderContent(g, contentSize.getWidth(), contentSize.getHeight());
		} catch (Exception e) {
			ThreadUtils.handleUncaughtException(e);
		}
	}

	/**
	 * Renders the content of the widget without the border, margin and padding specified in the style.
	 * <p>
	 * The given graphics context is translated and clipped according to the given bounds (the border, margin and
	 * padding are applied on this graphics context before).
	 *
	 * @param g
	 *            the graphics context where to render the content of the widget.
	 * @param contentWidth
	 *            the width of the content area.
	 * @param contentHeight
	 *            the height of the content area.
	 */
	protected abstract void renderContent(GraphicsContext g, int contentWidth, int contentHeight);

	/**
	 * Gets the parent of this widget or <code>null</code> if the widget is not in a hierarchy or if it is the root of
	 * its hierarchy.
	 *
	 * @return the parent of this widget or <code>null</code>.
	 */
	@Nullable
	public Container getParent() {
		Object parent = this.parent;
		if (parent instanceof Container) {
			return (Container) parent;
		} else {
			return null;
		}
	}

	/**
	 * Gets the desktop to which this widget has been added.
	 * <p>
	 * This method can only be called if this widget has been added to a desktop.
	 *
	 * @return the desktop to which this widgets has been added.
	 */
	public Desktop getDesktop() {
		Object parent = this.parent;
		assert (parent != null);
		if (parent instanceof Desktop) {
			return (Desktop) parent;
		} else {
			return ((Container) parent).getDesktop();
		}
	}

	@Nullable
	/* package */ Desktop getDesktopOrNull() {
		Object parent = this.parent;
		if (parent instanceof Desktop) {
			return (Desktop) parent;
		} else if (parent != null) {
			return ((Container) parent).getDesktopOrNull();
		} else {
			return null;
		}
	}

	/**
	 * Gets the current style of the widget.
	 * <p>
	 * This method should not be called before this widget is laid out.
	 *
	 * @return the current style of the widget.
	 * @see ej.mwt.stylesheet.Stylesheet#getStyle(Widget)
	 */
	public Style getStyle() {
		assert this.style != null;
		return this.style;
	}

	/**
	 * Sets the style of the widget.
	 *
	 * @param newStyle
	 *            the style to set.
	 */
	public void setStyle(Style newStyle) {
		this.style = newStyle;
	}

	/**
	 * Updates the style of this widget.
	 * <p>
	 * If the widget is not in a desktop, nothing is done.
	 */
	public void updateStyle() {
		Desktop desktop = getDesktopOrNull();
		if (desktop != null) {
			Style newStyle = desktop.getStylesheet().getStyle(this);
			setStyle(newStyle);
		}
	}

	/**
	 * Gets whether or not the widget has the given class selector.
	 *
	 * @param classSelector
	 *            the class selector to check.
	 * @return <code>true</code> if the widget has the given class selector, <code>false</code> otherwise.
	 */
	public boolean hasClassSelector(int classSelector) {
		return ArrayTools.getIndex(this.classSelectors, classSelector) != -1;
	}

	/**
	 * Adds a class selector.
	 *
	 * @param classSelector
	 *            the class selector to add.
	 * @throws NullPointerException
	 *             if the given class selector is <code>null</code>.
	 */
	public void addClassSelector(int classSelector) {
		this.classSelectors = ArrayTools.add(this.classSelectors, classSelector);
	}

	/**
	 * Removes a class selector.
	 *
	 * @param classSelector
	 *            the class selector to remove.
	 */
	public void removeClassSelector(int classSelector) {
		this.classSelectors = ArrayTools.remove(this.classSelectors, classSelector);
	}

	/**
	 * Sets the class selectors.
	 * <p>
	 * If there is already some class selectors, they are removed.
	 *
	 * @param classSelectors
	 *            the class selectors list to split.
	 */
	public void setClassSelectors(int[] classSelectors) {
		this.classSelectors = classSelectors.clone();
	}

	/**
	 * Removes all the class selectors.
	 */
	public void removeAllClassSelectors() {
		this.classSelectors = EMPTY_INT_ARRAY;
	}

	/**
	 * Gets whether or not the widget is in the given state.
	 *
	 * @param state
	 *            the state to check.
	 * @return <code>true</code> if the widget is in the given state, <code>false</code> otherwise.
	 */
	public boolean isInState(int state) {
		return false;
	}

	/**
	 * Gets a unique instance of rectangle. It is shared to avoid creating new objects for small operations in the
	 * MicroUI thread.
	 *
	 * @param x
	 *            the x coordinate to set.
	 * @param y
	 *            the y coordinate to set.
	 * @param width
	 *            the width to set.
	 * @param height
	 *            the height to set.
	 * @return the shared rectangle.
	 */
	/* package */ static Rectangle getSharedRectangle(int x, int y, int width, int height) {
		SHARED_RECTANGLE.setBounds(x, y, width, height);
		return SHARED_RECTANGLE;
	}

}
