/*
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector;

import ej.annotation.Nullable;
import ej.mwt.Widget;

/**
 * A class selector selects from a class. That means that the class of this selector is available in the widget classes.
 * <p>
 * Equivalent to <code>.class</code> selector in CSS. Its specificity is (0,0,1,0).
 *
 * @see Widget#hasClassSelector(int)
 * @see SelectorHelper
 */
public class ClassSelector implements Selector {

	private final int className;

	/**
	 * Creates a class selector.
	 *
	 * @param classSelector
	 *            the class
	 */
	public ClassSelector(int classSelector) {
		this.className = classSelector;
	}

	/**
	 * Gets the selected class.
	 * 
	 * @return the class
	 */
	public int getClassName() {
		return this.className;
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		// Check that the class of this selector is available in the given widget.
		return widget.hasClassSelector(this.className);
	}

	@Override
	public int getSpecificity() {
		return SelectorHelper.getSpecificity(0, 0, 1, 0);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			ClassSelector other = (ClassSelector) obj;
			return this.className == other.className;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return this.className;
	}

}
