/*
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline.border;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.drawing.ShapePainter;
import ej.drawing.ShapePainter.Cap;
import ej.microui.display.GraphicsContext;
import ej.mwt.style.outline.Outline;
import ej.mwt.util.Outlineable;
import ej.mwt.util.Size;

/**
 * Draws a uniform border with round corners.
 * <p>
 * May be associated with {@link ej.mwt.style.background.RoundedBackground} or used on its own.
 * <p>
 * The corner radius and thickness are stored as a <code>char</code> for heap optimization and therefore cannot exceed
 * <code>65535</code>.
 */
public class RoundedBorder implements Outline {

	private static final float ARC_ANGLE = 90.0f;
	private static final float START_ANGLE_TOPLEFT = 90.0f;
	private static final float START_ANGLE_BOTTOMLEFT = 180.0f;
	private static final float START_ANGLE_TOPRRIGHT = 0.0f;
	private static final float START_ANGLE_BOTTOMRIGHT = 270.0f;

	private static final int FADE = 1;
	private static final Cap CAP = Cap.NONE;

	private final int color;
	private final char cornerRadius;
	private final char thickness;

	/**
	 * Creates a rounded border specifying its attributes.
	 * <p>
	 * The given corner radius and thickness are clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param color
	 *            the color.
	 * @param cornerRadius
	 *            the corner radius.
	 * @param thickness
	 *            the border thickness.
	 */
	public RoundedBorder(int color, int cornerRadius, int thickness) {
		this.color = color;
		this.cornerRadius = (char) XMath.limit(cornerRadius, 0, Character.MAX_VALUE);
		this.thickness = (char) XMath.limit(thickness, 0, Character.MAX_VALUE);
	}

	/**
	 * Gets the color.
	 *
	 * @return the color.
	 */
	public int getColor() {
		return this.color;
	}

	/**
	 * Gets the corner radius.
	 *
	 * @return the corner radius.
	 */
	public int getCornerRadius() {
		return this.cornerRadius;
	}

	/**
	 * Gets the thickness.
	 *
	 * @return the thickness.
	 */
	public int getThickness() {
		return this.thickness;
	}

	@Override
	public void apply(Outlineable outlineable) {
		int thickness = this.thickness;
		outlineable.removeOutline(thickness, thickness, thickness, thickness);
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		// save background color
		boolean hasBackgroundColor = g.hasBackgroundColor();
		int backgroundColor = g.getBackgroundColor();

		// set color and background color
		g.setColor(this.color);
		g.removeBackgroundColor();

		// compute dimensions
		int width = size.getWidth();
		int height = size.getHeight();
		int thickness = this.thickness;
		int radius = Math.min(Math.max(this.cornerRadius, thickness + FADE), Math.min(width, height) >> 1);
		int diameter = (radius << 1);

		if (thickness > 0) {
			int actualThickness = thickness - FADE;
			int actualDiameter = diameter - actualThickness - FADE;
			int halfThickness = actualThickness >> 1;
			int oddThickness = thickness & 0x1;

			int shift = halfThickness - oddThickness + FADE;
			int oppositeShift = actualDiameter + shift;
			int xLeft = shift;
			int xRight = width - oppositeShift;
			int yTop = shift;
			int yBottom = height - oppositeShift;

			int clipX = g.getClipX();
			int clipY = g.getClipY();
			int clipWidth = g.getClipWidth();
			int clipHeight = g.getClipHeight();

			int xLeftLine = radius - oddThickness;
			int xRightLine = width - xLeftLine - 1;
			if (xLeftLine == xRightLine) {
				// drawThickFadedLine draws one more pixel at the end
				// - so with (left + 1 == right) two columns are drawn,
				// - but with (left == right) nothing is drawn.
				// Then the trick is to use the clip to draw only one column…
				g.intersectClip(xLeftLine, 0, 1, height);
				xRightLine++;
			}
			if (xLeftLine < xRightLine) {
				ShapePainter.drawThickFadedLine(g, xLeftLine, yTop, xRightLine, yTop, actualThickness, FADE, CAP, CAP);
				ShapePainter.drawThickFadedLine(g, xLeftLine, height - shift - FADE, xRightLine, height - shift - FADE,
						actualThickness, FADE, CAP, CAP);
			}
			g.setClip(clipX, clipY, clipWidth, clipHeight);
			int yTopLine = radius - oddThickness;
			int yBottomLine = height - yTopLine - 1;
			if (yTopLine == yBottomLine) {
				// drawThickFadedLine draws one more pixel at the end
				// - so with (top + 1 == bottom) two rows are drawn,
				// - but with (top == bottom) nothing is drawn.
				// Then the trick is to use the clip to draw only one row…
				g.intersectClip(0, yTopLine, width, 1);
				yBottomLine++;
			}
			if (yTopLine < yBottomLine) {
				ShapePainter.drawThickFadedLine(g, xLeft, yTopLine, xLeft, yBottomLine, actualThickness, FADE, CAP,
						CAP);
				ShapePainter.drawThickFadedLine(g, width - shift - FADE, yTopLine, width - shift - FADE, yBottomLine,
						actualThickness, FADE, CAP, CAP);
			}
			g.setClip(clipX, clipY, clipWidth, clipHeight);

			ShapePainter.drawThickFadedCircleArc(g, xLeft, yTop, actualDiameter, START_ANGLE_TOPLEFT, ARC_ANGLE,
					actualThickness, FADE, CAP, CAP);
			ShapePainter.drawThickFadedCircleArc(g, xLeft, yBottom, actualDiameter, START_ANGLE_BOTTOMLEFT, ARC_ANGLE,
					actualThickness, FADE, CAP, CAP);
			ShapePainter.drawThickFadedCircleArc(g, xRight, yTop, actualDiameter, START_ANGLE_TOPRRIGHT, ARC_ANGLE,
					actualThickness, FADE, CAP, CAP);
			ShapePainter.drawThickFadedCircleArc(g, xRight, yBottom, actualDiameter, START_ANGLE_BOTTOMRIGHT, ARC_ANGLE,
					actualThickness, FADE, CAP, CAP);
		}

		// apply outline
		size.removeOutline(thickness, thickness, thickness, thickness);
		g.translate(thickness, thickness);
		g.intersectClip(0, 0, size.getWidth(), size.getHeight());

		// restore background color
		if (hasBackgroundColor) {
			g.setBackgroundColor(backgroundColor);
		}
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			RoundedBorder border = (RoundedBorder) obj;
			return this.color == border.color && this.cornerRadius == border.cornerRadius
					&& this.thickness == border.thickness;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 17 * this.color + this.cornerRadius * this.thickness;
	}
}
