/*
 * Java
 *
 * Copyright 2017-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.rcommand.impl;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

/**
 * A chunked input stream is made of chunk of data with the following format: n byte count (int), n bytes of data.
 */
public class ChunkedInputStream extends InputStream {

	private final DataInputStream in;
	private byte[] chunk;
	private int position;
	private boolean eof;

	/**
	 * @param in
	 */
	public ChunkedInputStream(DataInputStream in) {
		this.in = in;
		this.chunk = new byte[0];

	}

	@Override
	public int available() throws IOException {
		return this.chunk.length - this.position;
	}

	@Override
	public long skip(long n) throws IOException {
		if (n <= 0 || this.eof) {
			return 0;
		}

		if (this.position >= this.chunk.length) {
			fillBuffer();
			if (this.eof) {
				return 0;
			}
		}

		long bytesSkipped = Math.min(this.chunk.length - this.position, n);
		this.position += bytesSkipped;
		return bytesSkipped;
	}

	@Override
	public int read() throws IOException {
		if (this.eof) {
			return -1;
		}

		if (this.position >= this.chunk.length) {
			fillBuffer();
			if (this.eof) {
				return -1;
			}
		}
		return this.chunk[this.position++] & 0xFF;
	}

	@Override
	public int read(byte[] b, int off, int len) throws IOException {
		if ((off | len | (off + len) | (b.length - (off + len))) < 0) {
			throw new IndexOutOfBoundsException();
		} else if (len == 0) {
			return 0;
		}

		if (this.eof) {
			return -1;
		}

		if (this.position >= this.chunk.length) {
			fillBuffer();
			if (this.eof) {
				return -1;
			}
		}

		int bytesRead = Math.min(this.chunk.length - this.position, len);
		System.arraycopy(this.chunk, this.position, b, off, bytesRead);
		this.position += bytesRead;
		return bytesRead;
	}

	private void fillBuffer() throws IOException {
		int chunkSize = this.in.readInt();

		if (chunkSize == 0) {
			this.eof = true;
			return;
		}

		if (this.chunk.length != chunkSize) {
			this.chunk = new byte[chunkSize];
		}
		chunkSize = this.in.read(this.chunk);
		if (chunkSize == -1) {
			throw new EOFException();
		}
		// might need more fills
		while (chunkSize < this.chunk.length) {
			int nbRead = this.in.read(this.chunk, chunkSize, this.chunk.length - chunkSize);
			if (nbRead == -1) {
				throw new EOFException();
			}
			chunkSize += nbRead;
		}
		if (chunkSize != this.chunk.length) {
			throw new IOException("chunkSize != this.chunk.length");
		}
		this.position = 0;
	}

	@Override
	public void close() throws IOException {
		this.eof = true;
	}
}
