/*
 * Copyright 2014-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package java.util.logging;

import java.lang.ref.WeakReference;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * There is one {@link LogManager} object in an application. It maintains properties about loggers and manages the hierarchy of loggers.
 */
public class LogManager {

	private class RootLogger extends Logger {

		protected RootLogger() {
			// The name of a root logger is ""
			super("", null);
			// Its level is given by the LogManager
			setLevel(getDefaultLevel());
		}
	}

	// Global and unique log manager
	private static final LogManager GLOBAL_LOG_MANAGER = newLogManager();

	// The root logger
	/* default */final Logger rootLogger;

	// All the available named Loggers
	private final Map<String, WeakReference<Logger>> namedLoggers;


	/**
	 * Protected constructor. There must be only one {@link LogManager} in an application.
	 */
	protected LogManager() {
		namedLoggers = new HashMap<>();

		rootLogger = new RootLogger();
		for (Handler handler : getDefaultHandlers()) {
			rootLogger.addHandler(handler);
		}
		addLogger(rootLogger);

	}

	private static LogManager newLogManager() {
		return new LogManager();
	}
	
	private Handler[] getDefaultHandlers() {
		String property = System.getProperty("handlers");
		// FIXME property may contain several class names

		Handler[] handlers = new Handler[0];

		if (property != null) {
			try {
				Class<?> classes = Class.forName(property);
				Handler handler = (Handler) classes.newInstance();
				handlers = new Handler[] { handler };
			} catch (ClassNotFoundException e) {
				e.printStackTrace();
			} catch (InstantiationException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			}
		}

		return handlers;
	}

	/**
	 * Read the configuration for the default level.
	 * 
	 * @return the default level for loggers
	 */
	private Level getDefaultLevel() {
		Level defaultLevel = Level.INFO; // see the reset() method in Java SE
		String property = System.getProperty(".level");

		Level[] predefined = { Level.OFF, Level.SEVERE, Level.WARNING, Level.INFO, Level.CONFIG, Level.FINE, Level.FINER, Level.FINEST, Level.ALL };
		for (Level level : predefined) {
			if (level.getName().equals(property)) {
				defaultLevel = level;
				break;
			}
		}

		// System.err.println("LogManager.getDefaultLevel() --> " + defaultLevel);
		return defaultLevel;
	}

	/**
	 * Returns the LogManager object.
	 * 
	 * @return the LogManager
	 */
	public static LogManager getLogManager() {
		return GLOBAL_LOG_MANAGER;
	}

	/**
	 * Add a named logger. This method does nothing if the logger had already been added.
	 * <p>
	 * The factory methods from {@link Logger} call this method for each Logger they created.
	 * 
	 * @param logger
	 *            the logger to register
	 * @return true if the logger was registered, false if a logger with the same already exists
	 * @throws NullPointerException
	 *             if the logger name is null
	 */
	public boolean addLogger(Logger logger) {
		WeakReference<Logger> weakLogger = namedLoggers.get(logger.getName());
		if (weakLogger != null && weakLogger.get() != null) {
			// This logger had already been added and the weak reference has not been garbage-collected
			return false;
		} else {
			// Add this logger (both case: we create a new entry OR we updated an existing entry that has been garbage-collected)
			String lowerCaseName = logger.getName().toLowerCase(); // can throw NullPointerException
			namedLoggers.put(lowerCaseName, new WeakReference<Logger>(logger));
			return true;
		}
	}

	/**
	 * Find a named logger.
	 * 
	 * @param name
	 *            name of the logger
	 * @return the matching logger or null if there is no match
	 */
	public Logger getLogger(String name) {
		String lowerCaseName = name.toLowerCase();
		WeakReference<Logger> weakLogger = namedLoggers.get(lowerCaseName);
		if (weakLogger == null) {
			return null; // no entry
		} else {
			return weakLogger.get(); // entry (that may be null)
		}
	}

	/**
	 * Get an enumeration of known logger names.
	 * 
	 * @return an enumeration of strings
	 */
	public Enumeration<String> getLoggerNames() {

		return new Enumeration<String>() {
			private final Iterator<String> iterator = namedLoggers.keySet().iterator();

			@Override
			public boolean hasMoreElements() {
				return iterator.hasNext();
			}

			@Override
			public String nextElement() {
				return iterator.next();
			}
		};
	}

}
