/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.image;

import java.awt.Color;
import java.nio.ByteBuffer;

import ej.microvg.image.LinearGradient.LinearGradientStop;
import ej.microvg.image.pathdata.TwoArraysPathData;

/**
 * The NemaGFX encoder.
 */
public class NemaImageGenerator implements ImageGenerator {

	private static final byte CMD_CLOSE = 0x00;
	private static final byte CMD_MOVE = 0x01;
	private static final byte CMD_MOVE_REL = 0x11;
	private static final byte CMD_LINE = 0x02;
	private static final byte CMD_LINE_REL = 0x12;
	private static final byte CMD_QUAD = 0x05;
	private static final byte CMD_QUAD_REL = 0x15;
	private static final byte CMD_CUBIC = 0x06;
	private static final byte CMD_CUBIC_REL = 0x16;

	private final TwoArraysPathData pathData;

	/**
	 * Creates the vector image generator that encodes the image in the Nema GPU compatible format.
	 *
	 * @param format
	 *            the path buffer output format expected by the application; this format is not used and the output
	 *            format is always {@link ImageGenerator.Format#VG_FP32}.
	 */
	public NemaImageGenerator(Format format) {
		this.pathData = new TwoArraysPathData();
	}

	@Override
	public int encodeCommand(Command command) {
		switch (command) {
		default:
		case COMMAND_CLOSE:
			return CMD_CLOSE;
		case COMMAND_MOVE:
			return CMD_MOVE;
		case COMMAND_MOVE_REL:
			return CMD_MOVE_REL;
		case COMMAND_LINE:
			return CMD_LINE;
		case COMMAND_LINE_REL:
			return CMD_LINE_REL;
		case COMMAND_QUAD:
			return CMD_QUAD;
		case COMMAND_QUAD_REL:
			return CMD_QUAD_REL;
		case COMMAND_CUBIC:
			return CMD_CUBIC;
		case COMMAND_CUBIC_REL:
			return CMD_CUBIC_REL;
		}
	}

	@Override
	public int encodeColor(Color color, float opacity) {
		int newColor = 0;
		newColor |= color.getRed();
		newColor |= color.getGreen() << 8;
		newColor |= color.getBlue() << 16;
		newColor |= (int) (opacity * 0xFF) << 24;
		return newColor;
	}

	@Override
	public byte[] encodePath(Path path) {
		TwoArraysPathData pathData = this.pathData;
		path.encode(pathData);

		byte[] commands = pathData.getPathCommands();
		byte[] params = pathData.getPathParams();
		int commandLength = commands.length;
		int paramLength = params.length;

		int size = 4 * Float.BYTES // bounding box
				+ 1 * Character.BYTES // command size
				+ 1 * Character.BYTES // param size
				+ paramLength + commandLength;

		ByteBuffer buffer = newByteBuffer(size);

		// header
		buffer.putFloat((float) path.getMinX()); // bounding box [0]
		buffer.putFloat((float) path.getMinY()); // bounding box [1]
		buffer.putFloat((float) path.getMaxX()); // bounding box [2]
		buffer.putFloat((float) path.getMaxY()); // bounding box [3]
		buffer.putChar((char) commandLength);
		buffer.putChar((char) (paramLength / pathData.getFormat().getUnitSize()));

		// data
		buffer.put(params, 0, paramLength);
		buffer.put(commands, 0, commandLength);

		return buffer.array();
	}

	@Override
	public byte[] encodeGradient(LinearGradient gradient) {
		int size = 4 * Float.BYTES + Integer.BYTES + gradient.getStops().length * (Float.BYTES + 4 * Float.BYTES);
		ByteBuffer buffer = newByteBuffer(size);

		LinearGradientStop[] stops = gradient.getStops();

		buffer.putFloat(gradient.getXStart());
		buffer.putFloat(gradient.getYStart());
		buffer.putFloat(gradient.getXEnd());
		buffer.putFloat(gradient.getYEnd());

		buffer.putInt(stops.length);
		for (LinearGradientStop stop : stops) {
			buffer.putFloat(stop.getPosition());
		}
		for (LinearGradientStop stop : stops) {
			Color color = stop.getColor();
			buffer.putFloat(color.getRed());
			buffer.putFloat(color.getGreen());
			buffer.putFloat(color.getBlue());
			buffer.putFloat(color.getAlpha());
		}

		return buffer.array();
	}
}
