/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.image;

import java.awt.Color;
import java.nio.ByteBuffer;

import ej.microvg.image.LinearGradient.LinearGradientStop;
import ej.microvg.image.pathdata.SingleArrayPathData;
import ej.microvg.image.pathdata.SingleArrayPathDataFP32;
import ej.microvg.image.pathdata.SingleArrayPathDataS16;
import ej.microvg.image.pathdata.SingleArrayPathDataS32;
import ej.microvg.image.pathdata.SingleArrayPathDataS8;

/**
 * The VGLite encoder.
 */
public class VgliteImageGenerator implements ImageGenerator {

	/**
	 * The path data formats.
	 */
	private static final int VG_LITE_S8 = 0;
	private static final int VG_LITE_S16 = 1;
	private static final int VG_LITE_S32 = 2;
	private static final int VG_LITE_FP32 = 3;

	/**
	 * The path commands.
	 */
	private static final int CMD_CLOSE = 0;
	private static final int CMD_MOVE = 2;
	private static final int CMD_MOVE_REL = 3;
	private static final int CMD_LINE = 4;
	private static final int CMD_LINE_REL = 5;
	private static final int CMD_QUAD = 6;
	private static final int CMD_QUAD_REL = 7;
	private static final int CMD_CUBIC = 8;
	private static final int CMD_CUBIC_REL = 9;

	/**
	 * The linear gradient.
	 *
	 * see vg_lite_gradient_t
	 *
	 * Do not encode the last element: vg_lite_buffer_t
	 */
	private static final float GRADIENT_SMALL_LENGTH = 0.1f; // Very small length to avoid scaling to 0.
	private static final int GRADIENT_MAX_STOPS = 255;
	private static final int GRADIENT_SCALE_NONE = 1; // Value to scale without any effect.
	private static final int GRADIENT_VLC_MAX_GRAD = 16; // VGLite lib value
	private static final int MAX_GRADIENT_STOP_COUNT = 16; // see VLC_MAX_GRAD in vg_lite.h
	private static final int GRADIENT_OFFSET_COLORS_SIZE = 0; // u32
	private static final int GRADIENT_OFFSET_XSTART = 1; // float
	private static final int GRADIENT_OFFSET_YSTART = 2; // float
	private static final int GRADIENT_OFFSET_ANGLE = 3; // float
	private static final int GRADIENT_OFFSET_LENGTH = 4; // float
	private static final int GRADIENT_OFFSET_COLORS_OFFSET = 5; // u32
	private static final int GRADIENT_OFFSET_POSITION_OFFSET = 6; // u32
	private static final int GRADIENT_HEADER_SIZE = 7;
	private static final int SIZE_VGLITE_GRADIENT = (1/* length */ + GRADIENT_VLC_MAX_GRAD * 2 /* colors and count */
			+ 9 /* matrix */) * Integer.BYTES;

	private final SingleArrayPathData pathData;

	/**
	 * Creates the vector image generator that encodes the image in the VGLite GPU compatible format.
	 *
	 * @param format
	 *            the path buffer output format expected by the application
	 */
	public VgliteImageGenerator(Format format) {
		this.pathData = newPathData(format);
	}

	@Override
	public int encodeCommand(Command command) {
		switch (command) {
		default:
		case COMMAND_CLOSE:
			return CMD_CLOSE;
		case COMMAND_MOVE:
			return CMD_MOVE;
		case COMMAND_MOVE_REL:
			return CMD_MOVE_REL;
		case COMMAND_LINE:
			return CMD_LINE;
		case COMMAND_LINE_REL:
			return CMD_LINE_REL;
		case COMMAND_QUAD:
			return CMD_QUAD;
		case COMMAND_QUAD_REL:
			return CMD_QUAD_REL;
		case COMMAND_CUBIC:
			return CMD_CUBIC;
		case COMMAND_CUBIC_REL:
			return CMD_CUBIC_REL;
		}
	}

	@Override
	public int encodeColor(Color color, float opacity) {
		int newColor = 0;

		newColor |= color.getRed() << 16;
		newColor |= color.getGreen() << 8;
		newColor |= color.getBlue();
		newColor |= (int) (opacity * 0xFF) << 24;

		return newColor;
	}

	@Override
	public byte[] encodePath(Path path) {
		SingleArrayPathData pathData = this.pathData;
		path.encode(pathData);

		byte[] data = pathData.getPath();

		int size = 4 * Float.BYTES // bounding box
				+ 1 * Integer.BYTES // command size
				+ 1 * Integer.BYTES // param size
				+ data.length // data
		;

		ByteBuffer buffer = newByteBuffer(size);

		// header
		buffer.putFloat((float) path.getMinX()); // bounding box [0]
		buffer.putFloat((float) path.getMinY()); // bounding box [1]
		buffer.putFloat((float) path.getMaxX()); // bounding box [2]
		buffer.putFloat((float) path.getMaxY()); // bounding box [3]
		buffer.putInt(data.length); // path data length
		buffer.put((byte) VgliteImageGenerator.convertFormat(pathData.getFormat())); // format
		buffer.put((byte) 0); // padding
		buffer.put((byte) 0); // padding
		buffer.put((byte) 0); // padding

		// data
		buffer.put(data);

		return buffer.array();
	}

	private static SingleArrayPathData newPathData(Format outputFormat) {
		switch (outputFormat) {
		case VG_S8:
			return new SingleArrayPathDataS8();
		case VG_S16:
			return new SingleArrayPathDataS16();
		case VG_S32:
			return new SingleArrayPathDataS32();
		case VG_FP32:
		default:
			return new SingleArrayPathDataFP32();
		}
	}

	/**
	 * Gets the VGLite data format from a {@link ImageGenerator.Format}.
	 *
	 * @param format
	 *            the format to convert.
	 * @return VGLite format representation.
	 */
	public static int convertFormat(Format format) {
		switch (format) {
		case VG_S8:
			return VG_LITE_S8;
		case VG_S16:
			return VG_LITE_S16;
		case VG_S32:
			return VG_LITE_S32;
		case VG_FP32:
		default:
			return VG_LITE_FP32;
		}
	}

	@Override
	public byte[] encodeGradient(LinearGradient gradient) {
		ByteBuffer buffer = newByteBuffer(SIZE_VGLITE_GRADIENT);

		int[] gradientHeader = getHeader(gradient);

		LinearGradientStop[] gradientStops = gradient.getStops();
		int padding = GRADIENT_VLC_MAX_GRAD - gradientStops.length;

		// colors
		for (LinearGradientStop gradientStop : gradientStops) {
			buffer.putInt(encodeColor(gradientStop.getColor(), gradientStop.getOpacity()));
		}
		for (int i = padding; --i >= 0;) {
			buffer.putInt(0);
		}

		buffer.putInt(gradientStops.length); // count

		// stops
		for (LinearGradientStop gradientStop : gradientStops) {
			buffer.putInt((int) (gradientStop.getPosition() * 255));
		}
		for (int i = padding; --i >= 0;) {
			buffer.putInt(0);
		}

		// calculate gradient's matrix (scale must be done after the rotate, otherwise
		// the rotation is eccentric (or put the same ratio for scale x and y)).
		float[] localMatrix = MatrixHelper.createIdentity();
		MatrixHelper.translate(localMatrix, Float.intBitsToFloat(gradientHeader[GRADIENT_OFFSET_XSTART]),
				Float.intBitsToFloat(gradientHeader[GRADIENT_OFFSET_YSTART]));
		MatrixHelper.rotate(localMatrix, Float.intBitsToFloat(gradientHeader[GRADIENT_OFFSET_ANGLE]));
		float length = Float.intBitsToFloat(gradientHeader[GRADIENT_OFFSET_LENGTH]);
		if (length == 0) {
			length = GRADIENT_SMALL_LENGTH;
		}
		MatrixHelper.scale(localMatrix, length / getGradientScaleSize(), GRADIENT_SCALE_NONE);

		// matrix
		for (float elem : localMatrix) {
			buffer.putFloat(elem);
		}

		return buffer.array();
	}

	private float getGradientScaleSize() {
		return GRADIENT_MAX_STOPS + 1f;
	}

	/**
	 * Gets the header of a gradient.
	 */
	@SuppressWarnings("nls")
	private static int[] getHeader(LinearGradient svgGradient) {
		float x1 = svgGradient.getXStart();
		float y1 = svgGradient.getYStart();
		float x2 = svgGradient.getXEnd();
		float y2 = svgGradient.getYEnd();

		float length = (float) Math.sqrt(Math.pow(x2 - x1, 2) + Math.pow(y2 - y1, 2));
		float angle = (float) Math.toDegrees(Math.atan2((y2 - y1), (x2 - x1)));

		LinearGradientStop[] gradientStops = svgGradient.getStops();
		int count = gradientStops.length;

		if (count > MAX_GRADIENT_STOP_COUNT) {
			throw new IllegalArgumentException("The number of stops(" + count + ") for gradient ([" + x1 + "," + y1
					+ "]->[" + x2 + "," + y2 + "]) exceed maximum VGLite value(" + MAX_GRADIENT_STOP_COUNT + ")");
		}

		int[] header = new int[GRADIENT_HEADER_SIZE];

		header[GRADIENT_OFFSET_COLORS_SIZE] = count;
		header[GRADIENT_OFFSET_XSTART] = Float.floatToRawIntBits(x1);
		header[GRADIENT_OFFSET_YSTART] = Float.floatToRawIntBits(y1);
		header[GRADIENT_OFFSET_ANGLE] = Float.floatToRawIntBits(angle);
		header[GRADIENT_OFFSET_LENGTH] = Float.floatToRawIntBits(length);
		header[GRADIENT_OFFSET_COLORS_OFFSET] = GRADIENT_HEADER_SIZE;
		header[GRADIENT_OFFSET_POSITION_OFFSET] = GRADIENT_HEADER_SIZE + count;

		return header;
	}

}
