/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.ResourceVectorImage;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.VectorImage;
import ej.microvg.VectorImageBuilder;

/**
 * Tests {@link ResourceVectorImage#close()} method.
 */
@SuppressWarnings("nls")
public class TestCloseImage {

	private static final float[] IDENTITY_MATRIX = new float[] { //
			1, 0, 0, 0, 0, // red
			0, 1, 0, 0, 0, // green
			0, 0, 1, 0, 0, // blue
			0, 0, 0, 1, 0 // alpha
	};

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Tests closing an image from a filtered one.
	 */
	@Test
	public static void testCloseFilterImage() {
		ResourceVectorImage resourceImage = createFilterImage();
		testCloseImage(resourceImage);
	}

	/**
	 * Tests closing an image from an image builder.
	 */
	@Test
	public static void testCloseBuilderImage() {
		ResourceVectorImage resourceImage = createBuilderImage();
		testCloseImage(resourceImage);
	}

	private static void testCloseImage(ResourceVectorImage resourceImage) {
		Assert.assertFalse("Image is closed", resourceImage.isClosed());
		resourceImage.close();
		Assert.assertTrue("Image is not closed", resourceImage.isClosed());
		resourceImage.close();
		Assert.assertTrue("Image is not closed (again)", resourceImage.isClosed());
	}

	/**
	 * Tests getting a closed image (from a filtered one) properties.
	 */
	@Test
	public static void testGetClosedFilterImageProperties() {
		ResourceVectorImage resourceImage = createFilterImage();
		testGetClosedImageProperties(resourceImage);
	}

	/**
	 * Tests getting a closed image (from an image builder) properties.
	 */
	@Test
	public static void testGetClosedBuilderImageProperties() {
		ResourceVectorImage resourceImage = createBuilderImage();
		testGetClosedImageProperties(resourceImage);
	}

	private static void testGetClosedImageProperties(ResourceVectorImage resourceImage) {
		resourceImage.close();
		Assert.assertEquals("Image width is wrong", 210.0f, resourceImage.getWidth(), 0.01f);
		Assert.assertEquals("Image height is wrong", 297.0f, resourceImage.getHeight(), 0.01f);
		Assert.assertEquals("Image duration is wrong", 0, resourceImage.getDuration());
	}

	/**
	 * Tests filtering a closed image (from a filtered one).
	 */
	@Test
	public static void testFilterClosedFilterImage() {
		ResourceVectorImage resourceImage = createFilterImage();
		testFilterClosedImage(resourceImage);
	}

	/**
	 * Tests filtering a closed image (from an image builder).
	 */
	@Test
	public static void testFilterClosedBuilderImage() {
		ResourceVectorImage resourceImage = createBuilderImage();
		testFilterClosedImage(resourceImage);
	}

	private static void testFilterClosedImage(ResourceVectorImage resourceImage) {

		resourceImage.close();
		boolean exception;
		try {
			resourceImage.filterImage(IDENTITY_MATRIX);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue("No exception when filtering a closed image", exception);
	}

	/**
	 * Tests drawing a closed image (from a filtered one).
	 */
	@Test
	public static void testDrawClosedFilterImage() {
		ResourceVectorImage resourceImage = createFilterImage();
		testDrawClosedImage(resourceImage);
	}

	/**
	 * Tests drawing a closed image (from an image builder).
	 */
	@Test
	public static void testDrawClosedBuilderImage() {
		ResourceVectorImage resourceImage = createBuilderImage();
		testDrawClosedImage(resourceImage);
	}

	private static void testDrawClosedImage(ResourceVectorImage resourceImage) {
		GraphicsContext g = Display.getDisplay().getGraphicsContext();
		resourceImage.close();
		boolean exception;
		try {
			VectorGraphicsPainter.drawImage(g, resourceImage, 0, 0);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue("No exception when drawing a closed image (1)", exception);
		try {
			VectorGraphicsPainter.drawImage(g, resourceImage, new Matrix());
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue("No exception when drawing a closed image (2)", exception);
		try {
			VectorGraphicsPainter.drawImage(g, resourceImage, new Matrix(), GraphicsContext.OPAQUE);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue("No exception when drawing a closed image (3)", exception);
		try {
			VectorGraphicsPainter.drawAnimatedImage(g, resourceImage, 0, 0, 0);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue("No exception when drawing a closed image (4)", exception);
		try {
			VectorGraphicsPainter.drawAnimatedImage(g, resourceImage, new Matrix(), 0);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue("No exception when drawing a closed image (5)", exception);
		try {
			VectorGraphicsPainter.drawAnimatedImage(g, resourceImage, new Matrix(), GraphicsContext.OPAQUE, 0);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue("No exception when drawing a closed image (6)", exception);
	}

	/**
	 * Tests allocating and closing a resource image a lot of times.
	 */
	@Test
	public static void testLeakImage() {
		for (int i = 0; i < 50000; i++) {
			createFilterImage().close();
			createBuilderImage().close();
		}
	}

	private static ResourceVectorImage createFilterImage() {
		VectorImage image = VectorImage.getImage("/com/microej/microvg/test/close_image.xml");
		return image.filterImage(IDENTITY_MATRIX);
	}

	private static ResourceVectorImage createBuilderImage() {
		VectorImageBuilder imageBuilder = new VectorImageBuilder(210, 297);
		Path path = new Path();
		path.lineTo(50, 0);
		path.lineTo(50, 100);
		path.lineTo(0, 100);
		path.close();
		imageBuilder.addPath(path, Colors.MAGENTA, FillType.EVEN_ODD);
		return imageBuilder.build();
	}

}
