/*
 * Copyright (c) 1997, 2012, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2017-2022 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. The Copyright holder(s) designates
 * this particular file as subject to the "Classpath" exception as provided
 * by the Copyright holder(s) in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact the Copyright holder(s) if you need additional information
 * or have any questions.
 * (if Oracle: 500 Oracle Parkway, Redwood Shores, CA 94065 USA or visit www.oracle.com)
 */

package javax.net.ssl;

import java.io.IOException;
import java.net.InetAddress;
import java.net.ServerSocket;

import ej.annotation.Nullable;

/**
 * This class extends <code>ServerSocket</code>s and provides secure server sockets using protocols such as the Secure
 * Sockets Layer (SSL) or Transport Layer Security (TLS) protocols.
 * <P>
 * Instances of this class are generally created using a <code>SSLServerSocketFactory</code>. The primary function of
 * <code>SSLServerSocket</code>s is to create <code>SSLSocket</code>s by <code>accept</code>ing connections.
 * <P>
 * <code>SSLServerSocket</code>s contain several pieces of state data which are inherited by the <code>SSLSocket</code>
 * at socket creation. These include the enabled cipher suites and protocols, whether client authentication is
 * necessary, and whether created sockets should begin handshaking in client or server mode. The state inherited by the
 * created <code>SSLSocket</code> can be overriden by calling the appropriate methods.
 *
 * @see java.net.ServerSocket
 * @see SSLSocket
 *
 * @since 1.4
 * @author David Brownell
 */
public abstract class SSLServerSocket extends ServerSocket {

	/**
	 * Used only by subclasses.
	 * <P>
	 * Create an unbound TCP server socket using the default authentication context.
	 *
	 * @throws IOException
	 *             if an I/O error occurs when creating the socket
	 */
	protected SSLServerSocket() throws IOException {
		super();
	}

	/**
	 * Used only by subclasses.
	 * <P>
	 * Create a TCP server socket on a port, using the default authentication context. The connection backlog defaults
	 * to fifty connections queued up before the system starts to reject new connection requests.
	 * <P>
	 * A port number of <code>0</code> creates a socket on any free port.
	 * <P>
	 * If there is a security manager, its <code>checkListen</code> method is called with the <code>port</code> argument
	 * as its argument to ensure the operation is allowed. This could result in a SecurityException.
	 *
	 * @param port
	 *            the port on which to listen
	 * @throws IOException
	 *             if an I/O error occurs when creating the socket
	 * @throws SecurityException
	 *             if a security manager exists and its <code>checkListen</code> method doesn't allow the operation.
	 * @throws IllegalArgumentException
	 *             if the port parameter is outside the specified range of valid port values, which is between 0 and
	 *             65535, inclusive.
	 */
	protected SSLServerSocket(int port) throws IOException {
		super(port);
	}

	/**
	 * Used only by subclasses.
	 * <P>
	 * Create a TCP server socket on a port, using the default authentication context and a specified backlog of
	 * connections.
	 * <P>
	 * A port number of <code>0</code> creates a socket on any free port.
	 * <P>
	 * The <code>backlog</code> argument is the requested maximum number of pending connections on the socket. Its exact
	 * semantics are implementation specific. In particular, an implementation may impose a maximum length or may choose
	 * to ignore the parameter altogther. The value provided should be greater than <code>0</code>. If it is less than
	 * or equal to <code>0</code>, then an implementation specific default will be used.
	 * <P>
	 * If there is a security manager, its <code>checkListen</code> method is called with the <code>port</code> argument
	 * as its argument to ensure the operation is allowed. This could result in a SecurityException.
	 *
	 * @param port
	 *            the port on which to listen
	 * @param backlog
	 *            requested maximum length of the queue of incoming connections.
	 * @throws IOException
	 *             if an I/O error occurs when creating the socket
	 * @throws SecurityException
	 *             if a security manager exists and its <code>checkListen</code> method doesn't allow the operation.
	 * @throws IllegalArgumentException
	 *             if the port parameter is outside the specified range of valid port values, which is between 0 and
	 *             65535, inclusive.
	 */
	protected SSLServerSocket(int port, int backlog) throws IOException {
		super(port, backlog);
	}

	/**
	 * Used only by subclasses.
	 * <P>
	 * Create a TCP server socket on a port, using the default authentication context and a specified backlog of
	 * connections as well as a particular specified network interface. This constructor is used on multihomed hosts,
	 * such as those used for firewalls or as routers, to control through which interface a network service is provided.
	 * <P>
	 * If there is a security manager, its <code>checkListen</code> method is called with the <code>port</code> argument
	 * as its argument to ensure the operation is allowed. This could result in a SecurityException.
	 * <P>
	 * A port number of <code>0</code> creates a socket on any free port.
	 * <P>
	 * The <code>backlog</code> argument is the requested maximum number of pending connections on the socket. Its exact
	 * semantics are implementation specific. In particular, an implementation may impose a maximum length or may choose
	 * to ignore the parameter altogther. The value provided should be greater than <code>0</code>. If it is less than
	 * or equal to <code>0</code>, then an implementation specific default will be used.
	 * <P>
	 * If <i>address</i> is null, it will default accepting connections on any/all local addresses.
	 *
	 * @param port
	 *            the port on which to listen
	 * @param backlog
	 *            requested maximum length of the queue of incoming connections.
	 * @param address
	 *            the address of the network interface through which connections will be accepted
	 * @throws IOException
	 *             if an I/O error occurs when creating the socket
	 * @throws SecurityException
	 *             if a security manager exists and its <code>checkListen</code> method doesn't allow the operation.
	 * @throws IllegalArgumentException
	 *             if the port parameter is outside the specified range of valid port values, which is between 0 and
	 *             65535, inclusive.
	 */
	protected SSLServerSocket(int port, int backlog, @Nullable InetAddress address) throws IOException {
		super(port, backlog, address);
	}

	/**
	 * Controls whether <code>accept</code>ed server-mode <code>SSLSockets</code> will be initially configured to
	 * <i>require</i> client authentication.
	 * <P>
	 * A socket's client authentication setting is one of the following:
	 * <ul>
	 * <li>client authentication required
	 * <li>client authentication requested
	 * <li>no client authentication desired
	 * </ul>
	 * <P>
	 * If the accepted socket's option is set and the client chooses not to provide authentication information about
	 * itself, <i>the negotiations will stop and the connection will be dropped</i>.
	 * <P>
	 * The initial inherited setting may be overridden by calling {@link SSLSocket#setNeedClientAuth(boolean)}
	 *
	 * @param need
	 *            set to true if client authentication is required, or false if no client authentication is desired.
	 * @see #getNeedClientAuth()
	 * @see #setUseClientMode(boolean)
	 */
	public abstract void setNeedClientAuth(boolean need);

	/**
	 * Returns true if client authentication will be <i>required</i> on newly <code>accept</code>ed server-mode
	 * <code>SSLSocket</code>s.
	 * <P>
	 * The initial inherited setting may be overridden by calling {@link SSLSocket#setNeedClientAuth(boolean)}
	 *
	 * @return true if client authentication is required, or false if no client authentication is desired.
	 * @see #setNeedClientAuth(boolean)
	 * @see #setUseClientMode(boolean)
	 */
	public abstract boolean getNeedClientAuth();

	/**
	 * Controls whether accepted connections are in the (default) SSL server mode, or the SSL client mode.
	 * <P>
	 * Servers normally authenticate themselves, and clients are not required to do so.
	 * <P>
	 * In rare cases, TCP servers need to act in the SSL client mode on newly accepted connections. For example, FTP
	 * clients acquire server sockets and listen there for reverse connections from the server. An FTP client would use
	 * an SSLServerSocket in "client" mode to accept the reverse connection while the FTP server uses an SSLSocket with
	 * "client" mode disabled to initiate the connection. During the resulting handshake, existing SSL sessions may be
	 * reused.
	 * <P>
	 * <code>SSLSocket</code>s returned from <code>accept()</code> inherit this setting.
	 *
	 * @param mode
	 *            true if newly accepted connections should use SSL client mode.
	 * @see #getUseClientMode()
	 */
	public abstract void setUseClientMode(boolean mode);

	/**
	 * Returns true if accepted connections will be in SSL client mode.
	 *
	 * @see #setUseClientMode(boolean)
	 * @return true if the connection should use SSL client mode.
	 */
	public abstract boolean getUseClientMode();
}
