/*
 * Copyright (c) 2001, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package sun.net.www.protocol.https;

import java.io.IOException;
import java.net.URL;

import sun.net.www.http.HttpClient;
import sun.net.www.protocol.http.HttpURLConnection;

/**
 * HTTPS URL connection support. We need this delegate because HttpsURLConnection is a subclass of
 * java.net.HttpURLConnection. We will avoid copying over the code from sun.net.www.protocol.http.HttpURLConnection by
 * having this class
 *
 */
public abstract class AbstractDelegateHttpsURLConnection extends HttpURLConnection {

	protected AbstractDelegateHttpsURLConnection(URL url, sun.net.www.protocol.http.Handler handler)
			throws IOException {
		super(url, handler);
	}

	protected abstract javax.net.ssl.SSLSocketFactory getSSLSocketFactory();

	/**
	 * No user application is able to call these routines, as no one should ever get access to an instance of
	 * DelegateHttpsURLConnection (sun.* or com.*)
	 */

	/**
	 * Create a new HttpClient object, bypassing the cache of HTTP client objects/connections.
	 *
	 * Note: this method is changed from protected to public because the com.sun.ssl.internal.www.protocol.https handler
	 * reuses this class for its actual implemantation
	 *
	 * @param url
	 *            the URL being accessed
	 */
	@Override
	public void setNewClient(URL url) throws IOException {
		HttpClient http = this.http = HttpsClient.New(getSSLSocketFactory(), url, this);
		((HttpsClient) http).afterConnect();
	}

	/**
	 * Used by subclass to access "connected" variable.
	 */
	public boolean isConnected() {
		return this.connected;
	}

	/**
	 * Used by subclass to access "connected" variable.
	 */
	public void setConnected(boolean conn) {
		this.connected = conn;
	}

	/**
	 * Implements the HTTP protocol handler's "connect" method, establishing an SSL connection to the server as
	 * necessary.
	 */
	@Override
	public void connect() throws IOException {
		if (this.connected) {
			return;
		}
		plainConnect();
		HttpClient http = this.http;
		assert http != null; // if plainConnect() above didn't throw an exception, http shouldn't be null
		((HttpsClient) http).afterConnect();
	}

	// will open new connection
	@Override
	protected HttpClient getNewHttpClient(URL url, int connectTimeout) throws IOException {
		return HttpsClient.New(getSSLSocketFactory(), url, connectTimeout, this);
	}

}
