/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.net;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;

/**
 * Socket input stream wrapper used to control net/ssl read call.
 */
public class SocketInputStream extends FilterInputStream {

	private final Socket socket;
	private static final Object lock = new Object();

	/**
	 * Creates a socket input stream instance.
	 *
	 * @param socket
	 * 		the socket.
	 * @param inputstream
	 * 		the underlying input stream.
	 * @throws IOException
	 * 		if an error occurs.
	 */
	public SocketInputStream(Socket socket, InputStream inputstream) throws IOException {
		super(inputstream);
		this.socket = socket;
	}

	@Override
	public int read() throws IOException {
		synchronized (lock) {
			ReadSocket readSocket = new ReadSocket(this.socket, 1);
			readSocket.onStart();
			try {
				int value = super.read();
				readSocket.onEnd(value);
				return value;
			} catch (IOException | IllegalArgumentException | SecurityException e) {
				readSocket.onEnd(-1);
				throw e;
			}
		}
	}

	@Override
	public int read(byte[] b, int off, int len) throws IOException {
		synchronized (lock) {
			ReadSocket readSocket = new ReadSocket(this.socket, len);
			int bytesAvailable = readSocket.onStart();
			try {
				int value = super.read(b, off, bytesAvailable);
				readSocket.onEnd(value);
				return value;
			} catch (IOException | IllegalArgumentException | SecurityException e) {
				readSocket.onEnd(-1);
				throw e;
			}
		}
	}

	@Override
	public void close() throws IOException {
		synchronized (lock) {
			boolean wasClosed = this.socket.isClosed();
			super.close();
			if (!wasClosed) {
				CloseSocket closeSocket = new CloseSocket(this.socket);
				closeSocket.onEnd(true);
			}
		}
	}
}
