/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.fs;

import com.microej.kf.util.module.SandboxedModule;
import com.microej.kf.util.module.SandboxedModuleHelper;
import ej.service.ServiceFactory;

import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * Action performed when setting the length of a RandomAccessFile.
 */
public class SetLengthRandomAccessFile {

	private final long nbBytes;

	/**
	 * Creates a write file action.
	 *
	 * @param file
	 * 		the file to write.
	 * @param newLength
	 * 		the new length to set.
	 * @throws java.io.IOException
	 * 		if an error occurs while accessing the file.
	 */
	public SetLengthRandomAccessFile(RandomAccessFile file, long newLength) throws IOException {
		this.nbBytes = newLength - file.length();
	}

	/**
	 * Gets the number of bytes to write.
	 *
	 * @return the number of bytes to write.
	 */
	public long getNbBytes() {
		return this.nbBytes;
	}

	/**
	 * Called when the execution of the action is about to start.
	 *
	 * This hook is used to check if the action is allowed to be performed.
	 *
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public void onStart() throws SecurityException {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getFileSystemResourceController().onStart(currentModule, this);
	}

	/**
	 * Called when the execution of the action is finished.
	 *
	 * This hook is used to update the current resources state of the module that performs the action.
	 *
	 * @param endWithSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public void onEnd(boolean endWithSuccess) {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getFileSystemResourceController().onEnd(currentModule, this, endWithSuccess);
	}
}
