/*
 * Java
 *
 * Copyright 2008-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp;

/**
 * An Image is a graphical element that can be displayed on a {@link Device}. An image must be obtained in a
 * platform-specific manner using {@link FrontPanel#newImage(int, int, int, boolean)} or
 * {@link FrontPanel#newImage(java.io.InputStream)}.
 */
public interface Image {

	/**
	 * Gets the width of the image.
	 *
	 * @return the width of the image.
	 */
	int getWidth();

	/**
	 * Gets the height of the image.
	 *
	 * @return the height of the image.
	 */
	int getHeight();

	/**
	 * Gets the color at the specified location.
	 *
	 * @param x
	 *            the x coordinate.
	 * @param y
	 *            the x coordinate.
	 * @return the color at the specified coordinate.
	 * @throws IndexOutOfBoundsException
	 *             if the specified coordinates are out of the image bounds.
	 */
	int readPixel(int x, int y);

	/**
	 * Gets all the pixels of the image.
	 *
	 * @param pixels
	 *            the array of pixels (its size must be greater equals to
	 *            <code>{@link #getWidth()} * {@link #getHeight()}</code>.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given pixels array is too small.
	 */
	void getPixels(int[] pixels);

	/**
	 * Draws a pixel at the specified coordinates and with the specified color.
	 *
	 * @param x
	 *            the x coordinate of the pixel.
	 * @param y
	 *            the y coordinate of the pixel.
	 * @param color
	 *            the color of the drawing.
	 */
	void drawPixel(int x, int y, int color);

	/**
	 * Draws a line with the specified boundary and the specified color.
	 *
	 * @param x1
	 *            the x coordinate of the first point.
	 * @param y1
	 *            the y coordinate of the first point.
	 * @param x2
	 *            the x coordinate of the second point.
	 * @param y2
	 *            the y coordinate of the second point.
	 * @param color
	 *            the color of the drawing.
	 */
	void drawLine(int x1, int y1, int x2, int y2, int color);

	/**
	 * Draws a rectangle with the specified bounds and the specified color.
	 *
	 * @param x
	 *            the x coordinate of the rectangle.
	 * @param y
	 *            the x coordinate of the rectangle.
	 * @param width
	 *            the width of the rectangle.
	 * @param height
	 *            the height of the rectangle.
	 * @param color
	 *            the color of the drawing.
	 */
	void drawRectangle(int x, int y, int width, int height, int color);

	/**
	 * Fills a rectangle with the specified bounds and the specified color.
	 *
	 * @param x
	 *            the x coordinate of the rectangle.
	 * @param y
	 *            the x coordinate of the rectangle.
	 * @param width
	 *            the width of the rectangle.
	 * @param height
	 *            the height of the rectangle.
	 * @param color
	 *            the color of the drawing.
	 */
	void fillRectangle(int x, int y, int width, int height, int color);

	/**
	 * Fills a transparent rectangle (force to turn this image as transparent image).
	 *
	 * @param x
	 *            the x coordinate of the rectangle.
	 * @param y
	 *            the x coordinate of the rectangle.
	 * @param width
	 *            the width of the rectangle.
	 * @param height
	 *            the height of the rectangle.
	 * @param argbColor
	 *            the rectangle color with the alpha level.
	 */
	void fillTransparentRectangle(int x, int y, int width, int height, int argbColor);

	/**
	 * Fills an oval bounded by the specified rectangle with the specified color.
	 *
	 * @param x
	 *            the x coordinate of the upper left corner of the oval to be filled.
	 * @param y
	 *            the y coordinate of the upper left corner of the oval to be filled.
	 * @param width
	 *            the width of the oval to be filled.
	 * @param height
	 *            the height of the oval to be filled.
	 * @param color
	 *            the color of the drawing.
	 */
	void fillOval(int x, int y, int width, int height, int color);

	/**
	 * Draws a string with default font at specified top-left position.
	 *
	 * @param s
	 *            the string to draw.
	 *
	 * @param x
	 *            the x coordinate of the upper left corner.
	 * @param y
	 *            the y coordinate of the upper left corner.
	 * @param textColor
	 *            the ARGB color of text.
	 * @param backColor
	 *            the ARGB color of background.
	 */
	void drawString(String s, int x, int y, int textColor, int backColor);

	/**
	 * Draws the given image in the image at position (0,0) (top-left). Source image may be deformed to fit destination
	 * rectangle: {@link #getWidth()} * {@link #getHeight()}.
	 *
	 * @param image
	 *            the image to draw.
	 */
	default void drawImage(Image image) {
		drawImage(image, 0, 0, getWidth(), getHeight());
	}

	/**
	 * Draws the given image in the image at given position. Source image may be deformed to fit destination rectangle.
	 * If the destination is outside the image bounds, nothing is drawn.
	 *
	 * @param image
	 *            the image to draw.
	 * @param dx
	 *            the x coordinate of the destination point.
	 * @param dy
	 *            the y coordinate of the destination point.
	 * @param dWidth
	 *            destination rectangle width.
	 * @param dHeight
	 *            destination rectangle height.
	 */
	default void drawImage(Image image, int dx, int dy, int dWidth, int dHeight) {
		drawImage(image, 0, 0, image.getWidth(), image.getHeight(), dx, dy, dWidth, dHeight);
	}

	/**
	 * Draws the part of the given image in the image at given position. Source image may be deformed to fit destination
	 * rectangle. If the area is outside the image bounds or if the destination is outside the image bounds, nothing is
	 * drawn.
	 *
	 * @param image
	 *            the image to draw.
	 * @param sx
	 *            the x coordinate of the image area to draw.
	 * @param sy
	 *            the y coordinate of the image area to draw.
	 * @param sWidth
	 *            the width of the image area to draw.
	 * @param sHeight
	 *            the height of the image area to draw.
	 * @param dx
	 *            the x coordinate of the destination point.
	 * @param dy
	 *            the y coordinate of the destination point.
	 * @param dWidth
	 *            destination rectangle width.
	 * @param dHeight
	 *            destination rectangle height.
	 */
	void drawImage(Image image, int sx, int sy, int sWidth, int sHeight, int dx, int dy, int dWidth, int dHeight);

	/**
	 * Draws the specified area of the given image at the specified destination using alpha blending. Source image may
	 * be deformed to fit destination rectangle.
	 *
	 * @param image
	 *            the image to draw.
	 * @param alpha
	 *            the alpha blending to apply during drawing (0 <= alpha <= 0xff).
	 * @param sx
	 *            the top-left X coordinate of the area to draw.
	 * @param sy
	 *            the top-left Y coordinate of the area to draw.
	 * @param sWidth
	 *            the width of the area to draw.
	 * @param sHeight
	 *            the height of the area to draw.
	 * @param dx
	 *            the top-left X coordinate of the destination.
	 * @param dy
	 *            the top-left Y coordinate of the destination.
	 * @param dWidth
	 *            destination rectangle width.
	 * @param dHeight
	 *            destination rectangle height.
	 */
	void drawTransparentImage(Image image, byte alpha, int sx, int sy, int sWidth, int sHeight, int dx, int dy,
			int dWidth, int dHeight);

	/**
	 * Crop this image keeping only non-transparent pixels from given image mask.
	 * <p>
	 * If mask is null, nothing is cropped. When not null, the mask must have the same size than this image.
	 *
	 * @param mask
	 *            the reference image.
	 * @throws IllegalArgumentException
	 *             if the given mask has not the same size than this image.
	 */
	void crop(Image mask) throws IllegalArgumentException;

	/**
	 * Sets all the pixels of the image.
	 *
	 * @param pixels
	 *            the array of pixels (its size must be greater equals to
	 *            <code>{@link #getWidth()} * {@link #getHeight()}</code>
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given pixels array is too small.
	 */
	void setPixels(int[] pixels);

	/**
	 * All drawings in transparent mode manage alpha channel for the given colors.
	 *
	 * @param transparent
	 *            <code>true</code> to set transparent mode, <code>false</code> otherwise.
	 */
	void setTransparentMode(boolean transparent);

	/**
	 * Gets a transparent image of the image.
	 *
	 * @param alpha
	 *            the opacity level (0xff is opaque, 0x00 is transparent).
	 * @return the transparent image.
	 */
	Image getTransparentImage(byte alpha);

	/**
	 * Disposes the image and its resources.
	 */
	void dispose();
}
