/*
 * Copyright (C) 2014 The Android Open Source Project
 * Copyright (C) 2015-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.net;

import java.util.Arrays;

import ej.annotation.Nullable;
import ej.basictool.ArrayTools;

/**
 * This class represents the capabilities of a network. This is used both to specify needs to
 * {@link ConnectivityManager} and when inspecting a network.
 *
 * Note that this replaces the old ConnectivityManager#TYPE_MOBILE method of network selection. Rather than indicate a
 * need for Wi-Fi because an application needs high bandwidth and risk obselence when a new, fast network appears (like
 * LTE), the application should specify it needs high bandwidth. Similarly if an application needs an unmetered network
 * for a bulk transfer it can specify that rather than assuming all cellular based connections are metered and all Wi-Fi
 * based connections are not.
 */
public class NetworkCapabilities {

	/**
	 * Indicates that this network should be able to reach the internet.
	 */
	public static final int NET_CAPABILITY_INTERNET = 0x0000000c;

	private int[] capabilities;

	/**
	 * Instantiates a {@link NetworkCapabilities} without capabilities.
	 */
	public NetworkCapabilities() {
		this(new int[0]);
	}

	/**
	 * Instantiates a {@link NetworkCapabilities}.
	 *
	 * @param capabilities
	 *            capabilities;
	 */
	public NetworkCapabilities(int[] capabilities) {
		this.capabilities = ArrayTools.copy(capabilities);
	}

	/**
	 * Tests for the presence of a capability on this instance.
	 *
	 * @param capability
	 *            Value is {@link NetworkCapabilities}
	 * @return <code>true</code> if set on this instance.
	 */
	public boolean hasCapability(int capability) {
		boolean contain = false;
		for (int i : this.capabilities) {
			if (i == capability) {
				contain = true;
				break;
			}
		}
		return contain;
	}

	/**
	 * Sets the capabilities.
	 *
	 * @param capabilities
	 *            the capabilities to set.
	 */
	protected void setCapabilities(int[] capabilities) {
		this.capabilities = ArrayTools.copy(capabilities);
	}

	/**
	 * Gets the capabilities.
	 *
	 * @return the capabilities.
	 */
	protected int[] getCapabilities() {
		return this.capabilities;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + Arrays.hashCode(this.capabilities);
		return result;
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		NetworkCapabilities other = (NetworkCapabilities) obj;
		if (!Arrays.equals(this.capabilities, other.capabilities)) {
			return false;
		}
		return true;
	}
}
