/*
 * Java
 *
 * Copyright 2016-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.ecom.network;

import java.io.IOException;
import java.net.NetworkInterface;

/**
 * A manager for the network interfaces.
 *
 * It allows to get the current IP configuration of a network interface and to
 * reconfigure it. <br>
 * It allows to enable/disable and start/stop an interface. <br>
 * - Enable/disable is simply a logical state of the interface managed by
 * software.<br>
 * - Start/stop is more hardware-related. It may physically turn on/off the
 * hardware module and it is intended to manage power consumption.<br>
 * <br>
 * <br>
 * Scenario example: a system uses Ethernet as its main interface and 3G as its
 * backup interface. In normal mode, Ethernet is started and enabled while 3G is
 * stopped (and hence disabled) and communications all go through the Ethernet
 * interface. When Ethernet loses Internet connection, it is disabled and 3G is
 * started. Communications will then go through the 3G interface. When Ethernet
 * gets Internet connection back, it is re-enabled, 3G is stopped and the system
 * goes back to normal mode.
 */
public class NetworkInterfaceManager {

	/**
	 * Sets the IP configuration of a network interface.
	 *
	 * The configuration may be sparse. Only not-<code>null</code> and relevant
	 * fields are applied. For instance, static IP address is ignored if DHCP is to
	 * be used.
	 *
	 * When the configuration is applied, be aware that:
	 * <ul>
	 * <li>current communications may be interrupted, raising {@link IOException},
	 * <li>the hardware module associated to the network interface may reboot,
	 * making the interface not usable for a few seconds,
	 * <li>it may take time to acquire an IP address if the configuration requires
	 * the use of DHCP, making the interface not usable for a few seconds.
	 * </ul>
	 *
	 * Note that some DNS server addresses may be silently ignored if the underlying
	 * module doesn't support several DNS servers.
	 *
	 * @param netif
	 *            the network interface.
	 * @param conf
	 *            the network interface IP configuration.
	 * @throws IOException
	 *             if it is not possible to reconfigure the interface.
	 * @throws IllegalArgumentException
	 *             if netif or conf is <code>null</code>.
	 */
	public static void configure(NetworkInterface netif, IPConfiguration conf) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Disables a network interface (software side).
	 *
	 * A disabled network interface is no longer available for communications.
	 * Current communications may be interrupted, raising {@link IOException}.
	 *
	 * @param netif
	 *            the network interface to disable.
	 * @throws IllegalStateException
	 *             if the network interface is currently stopped.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if netif is <code>null</code>.
	 */
	public static void disable(NetworkInterface netif) throws IllegalStateException, IOException {
		throw new RuntimeException();
	}

	/**
	 * Enables a network interface (software side).
	 *
	 * An enabled network interface is available for communications, assuming that
	 * {@link #start(NetworkInterface)} has been called beforehand.
	 *
	 * @param netif
	 *            the network interface to enable.
	 * @throws IllegalStateException
	 *             if the interface is currently stopped.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if netif is <code>null</code>.
	 */
	public static void enable(NetworkInterface netif) throws IllegalStateException, IOException {
		throw new RuntimeException();
	}

	/**
	 * Gets the current IP configuration of a network interface.
	 *
	 * @param netif
	 *            the network interface.
	 * @return the current IP configuration of the network interface.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if netif is <code>null</code>.
	 */
	public static IPConfiguration getIPConfiguration(NetworkInterface netif) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Tells whether a network interface is enabled or not.
	 *
	 * @param netif
	 *            the network interface.
	 * @return <code>true</code> if enabled, <code>false</code> otherwise
	 * @throws IllegalArgumentException
	 *             if netif is <code>null</code>.
	 */
	public static boolean isEnabled(NetworkInterface netif) {
		throw new RuntimeException();
	}

	/**
	 * Tells whether a network interface is started or not.
	 *
	 * @param netif
	 *            the network interface.
	 * @return <code>true</code> if started, <code>false</code> otherwise.
	 * @throws IllegalArgumentException
	 *             if netif is <code>null</code>.
	 */
	public static boolean isStarted(NetworkInterface netif) {
		throw new RuntimeException();
	}

	/**
	 * Starts a network interface (hardware side).
	 *
	 * Switch on the network interface. The network interface is not enabled,
	 * {@link #enable(NetworkInterface)} has to be called after this method.
	 *
	 * @param netif
	 *            the network interface to start.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if netif is <code>null</code>.
	 */
	public static void start(NetworkInterface netif) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Stops a network interface (hardware side).
	 *
	 * Stopping a network interface will also disable it. Current communications may
	 * be interrupted, raising IOExceptions.
	 *
	 * @param netif
	 *            the interface to stop.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if netif is <code>null</code>.
	 */
	public static void stop(NetworkInterface netif) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Set a network interface as default. This default network interface will be
	 * used for all communications which do not specified a targeted network
	 * interface.
	 *
	 * @param netif the network interface to use as the default one.
	 * @throws IOException              if an I/O error occurs.
	 * @throws IllegalArgumentException if netif is <code>null</code>.
	 */
	public static void setDefault(NetworkInterface netif) throws IOException {
		throw new RuntimeException();
	}
}
