/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

/**
 * The <code>BluetoothConnection</code> class represents a connection with a remote Bluetooth device.
 * <p>
 * Connections may be retrieved by listening to
 * {@link ej.bluetooth.listeners.ConnectionListener#onConnected(BluetoothConnection) connection events} while
 * {@link BluetoothAdapter#connect(BluetoothAddress) connecting} to a device or while
 * {@link BluetoothAdapter#startAdvertising(byte[]) advertising}. The GAP events related to a device may be handled by
 * {@link BluetoothAdapter#setConnectionListener setting the connection listener} of the adapter.
 * <p>
 * This class provides a method to get the address of the remote device. It also allows to disconnect to the device, to
 * manage pairing and to discover the services that the device provides.
 */
public class BluetoothConnection {

	/**
	 * Private constructor.
	 */
	private BluetoothConnection() {
		throw new RuntimeException();
	}

	/**
	 * Returns the Bluetooth address of the device.
	 *
	 * @return the Bluetooth address of the device.
	 */
	public BluetoothAddress getAddress() {
		throw new RuntimeException();
	}

	/**
	 * Initiates a disconnection from the device.
	 *
	 * @return true if disconnection was initiated successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onDisconnected(BluetoothConnection)
	 */
	public boolean disconnect() {
		throw new RuntimeException();
	}

	/**
	 * Initiates pairing with the device.
	 * <p>
	 * This operation may only be performed while the adapter is still connected to the device.
	 *
	 * @return true if pairing was initiated successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onPairCompleted(BluetoothConnection, boolean)
	 * @see ej.bluetooth.listeners.ConnectionListener#onPasskeyRequest(BluetoothConnection)
	 */
	public boolean sendPairRequest() {
		throw new RuntimeException();
	}

	/**
	 * Replies to the pairing request of the device.
	 * <p>
	 * This operation may only be performed after receiving a pairing request.
	 *
	 * @param accept
	 *            true if the pairing request should be accepted, false otherwise.
	 * @return true if the reply was sent successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onPairCompleted(BluetoothConnection, boolean)
	 * @see ej.bluetooth.listeners.ConnectionListener#onPasskeyGenerated(BluetoothConnection, int)
	 */
	public boolean sendPairResponse(boolean accept) {
		throw new RuntimeException();
	}

	/**
	 * Replies to the passkey request of the device.
	 * <p>
	 * This operation may only be performed after receiving a passkey request.
	 *
	 * @param accept
	 *            true if the passkey request should be accepted, false otherwise.
	 * @param passkey
	 *            the passkey value (6-digits PIN code). This parameter is used only if the request is accepted.
	 * @return true if the reply was sent successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onPairCompleted(BluetoothConnection, boolean)
	 */
	public boolean sendPasskeyResponse(boolean accept, int passkey) {
		throw new RuntimeException();
	}

	/**
	 * Discovers all the services provided by the device.
	 * <p>
	 * This operation may only be performed while the adapter is still connected to the device. In order to discover
	 * specific services only, use {@link #discoverService(BluetoothUuid)} instead.
	 *
	 * @return true if service discovery was initiated successfully, false otherwise.
	 * @see #discoverService(BluetoothUuid)
	 * @see ej.bluetooth.listeners.ConnectionListener#onDiscoveryResult(BluetoothConnection, BluetoothService)
	 * @see ej.bluetooth.listeners.ConnectionListener#onDiscoveryCompleted(BluetoothConnection)
	 */
	public boolean discoverServices() {
		throw new RuntimeException();
	}

	/**
	 * Discovers the service with the given UUID provided by the device.
	 * <p>
	 * This operation may only be performed while the adapter is still connected to the device. In order to discover all
	 * services, use {@link #discoverServices()} instead.
	 *
	 * @param uuid
	 *            the UUID of the service to discover.
	 * @return true if service discovery was initiated successfully, false otherwise.
	 * @see #discoverServices()
	 * @see ej.bluetooth.listeners.ConnectionListener#onDiscoveryResult(BluetoothConnection, BluetoothService)
	 * @see ej.bluetooth.listeners.ConnectionListener#onDiscoveryCompleted(BluetoothConnection)
	 */
	public boolean discoverService(BluetoothUuid uuid) {
		throw new RuntimeException();
	}

	/**
	 * Sends a read request on the given attribute.
	 *
	 * @param attribute
	 *            the attribute to send a read request on.
	 * @return true if the read request was sent successfully.
	 * @throws IllegalArgumentException
	 *             if the attribute is not a remote attribute.
	 * @throws IllegalArgumentException
	 *             if the attribute does not belong to the device.
	 * @see ej.bluetooth.listeners.RemoteServiceListener#onReadCompleted(BluetoothConnection, BluetoothAttribute, byte,
	 *      byte[])
	 */
	public boolean sendReadRequest(BluetoothAttribute attribute) {
		throw new RuntimeException();
	}

	/**
	 * Sends a write request on the given attribute.
	 *
	 * @param attribute
	 *            the attribute to send a write request on.
	 * @param value
	 *            the value to write.
	 * @return true if the write request was sent successfully.
	 * @throws IllegalArgumentException
	 *             if the attribute is not a remote attribute.
	 * @throws IllegalArgumentException
	 *             if the attribute does not belong to the device.
	 * @see ej.bluetooth.listeners.RemoteServiceListener#onWriteCompleted(BluetoothConnection, BluetoothAttribute, byte)
	 */
	public boolean sendWriteRequest(BluetoothAttribute attribute, byte[] value) {
		throw new RuntimeException();
	}

	/**
	 * Sends the response to a read request sent by a device on the given attribute.
	 *
	 * @param attribute
	 *            the attribute on which the request was sent.
	 * @param status
	 *            the status to send back (see {@link BluetoothStatus}).
	 * @param value
	 *            the value to send back.
	 * @return true if the read response was sent successfully.
	 * @throws IllegalArgumentException
	 *             if the attribute is not a local attribute.
	 */
	public boolean sendReadResponse(BluetoothAttribute attribute, byte status, byte[] value) {
		throw new RuntimeException();
	}

	/**
	 * Sends the response to a write request sent by a device on the given attribute.
	 *
	 * @param attribute
	 *            the attribute on which the request was sent.
	 * @param status
	 *            the status to send back (see {@link BluetoothStatus}).
	 * @return true if the write response was sent successfully.
	 * @throws IllegalArgumentException
	 *             if the attribute is not a local attribute.
	 */
	public boolean sendWriteResponse(BluetoothAttribute attribute, byte status) {
		throw new RuntimeException();
	}

	/**
	 * Sends the response to a prepare write request sent by a device on the given attribute.
	 *
	 * @param attribute
	 *            the attribute on which the request was sent.
	 * @param status
	 *            the status to send back (see {@link BluetoothStatus}).
	 * @param value
	 *            the value of the prepare write request.
	 * @param offset
	 *            the offset of the prepare write request.
	 * @return true if the prepare write response was sent successfully.
	 * @throws IllegalArgumentException
	 *             if the attribute is not a local attribute.
	 */
	public boolean sendPrepareWriteResponse(BluetoothAttribute attribute, byte status, byte[] value, int offset) {
		throw new RuntimeException();
	}

	/**
	 * Sends the response to an execute write request sent by a device on the given attribute.
	 *
	 * @param attribute
	 *            the attribute on which the request was sent.
	 * @param status
	 *            the status to send back (see {@link BluetoothStatus}).
	 * @return true if the execute write response was sent successfully.
	 * @throws IllegalArgumentException
	 *             if the attribute is not a local attribute.
	 */
	public boolean sendExecuteWriteResponse(BluetoothAttribute attribute, byte status) {
		throw new RuntimeException();
	}

	/**
	 * Sends a notification or indication on the given characteristic to a device.
	 *
	 * @param characteristic
	 *            the characteristic to send a notification on.
	 * @param value
	 *            the value to send.
	 * @param confirm
	 *            true to request a confirmation (GATT indication), false otherwise (GATT notification).
	 * @return true if the notification was sent successfully.
	 * @throws IllegalArgumentException
	 *             if the characteristic is not a local characteristic.
	 * @see ej.bluetooth.listeners.LocalServiceListener#onNotificationSent(BluetoothConnection, BluetoothCharacteristic,
	 *      boolean)
	 */
	public boolean sendNotification(BluetoothCharacteristic characteristic, byte[] value, boolean confirm) {
		throw new RuntimeException();
	}
}
