/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.microej.microvg.test.FirstFont.FirstFontCharacter;
import com.microej.microvg.test.TestUtilities.VEEPort;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.DisplayUtilities;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests the drawing of strings.
 */
public class TestDrawString {

	private static final float DEFAULT_LETTER_SPACING = 0f;
	private static final int PADDING = 1;
	private static final String STRING_SQUARE_SQUARE_SQUARE = "AAA";
	private static final String STRING_SQUARE_BASELINE_SQUARE = "ABA";
	private static final String STRING_SQUARE_MID_SQUARE = "ALA";
	private static final String STRING_SQUARE = "A";
	private static final String STRING_NO_GLYPH = "@";
	private static final String STRING_MID_SQUARE = "LA";
	private static final String STRING_OUTLEFT_SQUARE = "MA";
	private static final String STRING_LEFTINFLEXIONPOINT = "PA";
	private static final String STRING_LEFTSPACE = " A";
	private static final String STRING_RIGHTSPACE = "A ";
	private static final String STRING_SPACEINSIDE = "A A";
	private static final String STRING_SPACE = " ";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing a string with different colors.
	 */
	@Test
	public void testStringColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;
		int fontSize = 48;
		int baselinePosition = (int) font.getBaselinePosition(fontSize);
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		int[] colorSet = { Colors.WHITE, Colors.CYAN, Colors.RED, Colors.GREEN, Colors.MAGENTA };
		for (int color : colorSet) {
			g.setColor(color);
			VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
			display.flush();

			// Check first character : full box filled
			int advance = 0;
			int firstCharWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
			TestUtilities.checkArea("color first character", color, x, y, firstCharWidth, fontSize, PADDING);
			advance += firstCharWidth;

			// Check second character: above baseline filled, nothing below
			// above baseline is color
			int secondCharWidth = (int) font.measureStringWidth(string.substring(1, 2), fontSize);
			TestUtilities.checkArea("color second character above baseline", color, x + advance, y, secondCharWidth,
					baselinePosition, PADDING);

			// below baseline is black
			TestUtilities.checkArea("color second character below baseline", TestUtilities.BACKGROUND_COLOR, x + advance,
					y + baselinePosition, secondCharWidth, fontSize - baselinePosition, PADDING);
			advance += secondCharWidth;

			// Check 3rd character: full box filled
			int thirdCharWidth = (int) font.measureStringWidth(string.substring(2, 3), fontSize);
			TestUtilities.checkArea("color third character", color, x + advance, y, thirdCharWidth, fontSize, PADDING);
			int stringWidth = advance + thirdCharWidth;

			// Check no drawing outside
			TestUtilities.checkPeripheralArea("color", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize, 50, 3);

			// Clear screen
			TestUtilities.clearScreen();
		}
	}

	/**
	 * Test drawing a string with different font sizes.
	 */
	@Test
	public void testStringSize() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;
		int color = Colors.BLUE;

		int x = display.getWidth() / 5;
		int y = display.getHeight() / 5;

		int fontSizes[] = { 10, 18, 24, 28, 36, 50, 70 };
		for (int fontSize : fontSizes) {
			// Clean before drawstring
			TestUtilities.clearScreen();

			g.setColor(color);
			VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
			display.flush();

			// Check first character : full box filled
			int baselinePosition = (int) font.getBaselinePosition(fontSize);
			int advance = 0;
			int firstCharWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
			TestUtilities.checkArea("size first character", color, x, y, firstCharWidth, fontSize, PADDING);
			advance += firstCharWidth;

			// Check second character: above baseline filled, nothing below
			// above baseline is color
			int secondCharWidth = (int) font.measureStringWidth(string.substring(1, 2), fontSize);
			TestUtilities.checkArea("size second character above baseline", color, x + advance, y, secondCharWidth,
					baselinePosition, PADDING);

			// below baseline is black
			TestUtilities.checkArea("size second character below baseline", TestUtilities.BACKGROUND_COLOR, x + advance,
					y + baselinePosition, secondCharWidth, fontSize - baselinePosition, PADDING);
			advance += secondCharWidth;

			// Check 3rd character: full box filled
			int thirdCharWidth = (int) font.measureStringWidth(string.substring(2, 3), fontSize);
			TestUtilities.checkArea("size third character", color, x + advance, y, thirdCharWidth, fontSize, PADDING);
			int stringWidth = advance + thirdCharWidth;

			// Check no drawing outside
			TestUtilities.checkPeripheralArea("size", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize, 50, 3);

			// Clear screen
			TestUtilities.clearScreen();
		}
	}

	/**
	 * Test drawing a string with different alpha.
	 *
	 * <p>
	 * The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int fontSize = 48;
		String string = STRING_SQUARE_BASELINE_SQUARE;
		int color = Colors.WHITE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		int[] alphaSet = { 0x0, 0x40, 0x80, 0xC0, 0xFF };
		for (int alpha : alphaSet) {

			g.setColor(color);
			VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, alpha, BlendMode.SRC_OVER,
					DEFAULT_LETTER_SPACING);
			display.flush();

			int expectedColor = TestUtilities.blendColors(color, display.getDisplayColor(TestUtilities.BACKGROUND_COLOR), alpha);

			TestUtilities.check("alpha first character", x, y, g, expectedColor);
			TestUtilities.check("alpha second character",
					(int) (x + font.measureStringWidth(string.substring(0, 1), fontSize)), y, g, expectedColor);
			TestUtilities.check("alpha third character",
					(int) (x + font.measureStringWidth(string.substring(0, 2), fontSize)), y, g, expectedColor);

			// Clear screen to blend over black
			TestUtilities.clearScreen();
		}
	}

	/**
	 * Test drawing a string with a translation matrix.
	 *
	 * <p>
	 * The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringMatrixTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int fontSize = 48;
		String string = STRING_SQUARE_SQUARE_SQUARE;
		int color = Colors.WHITE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);
		display.flush();

		// check all characters
		TestUtilities.check("translate first character", x, y, g, color);
		TestUtilities.check("translate second character",
				(int) (x + font.measureStringWidth(string.substring(0, 1), fontSize)), y, g, color);
		TestUtilities.check("translate third character",
				(int) (x + font.measureStringWidth(string.substring(0, 2), fontSize)), y, g, color);

		// Check no drawing outside
		int stringWidth = (int) font.measureStringWidth(string, fontSize);
		TestUtilities.checkPeripheralArea("translate", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize, 50, 3);
	}

	/**
	 * Test drawing a string with a rotation matrix.
	 *
	 * <p>
	 * The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringMatrixRotate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int fontSize = 48;
		String string = STRING_SQUARE_SQUARE_SQUARE;
		int color = Colors.WHITE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		matrix.preRotate(90);

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);
		display.flush();

		// check all characters
		TestUtilities.check("rotate first character", x - fontSize / 2, y, g, color);
		TestUtilities.check("rotate second character", x - fontSize / 2,
				(int) (y + font.measureStringWidth(string.substring(0, 1), fontSize)), g, color);
		TestUtilities.check("rotate third character", x - fontSize / 2,
				(int) (y + font.measureStringWidth(string.substring(0, 2), fontSize)), g, color);

		// Check no drawing outside
		int stringWidth = (int) font.measureStringWidth(string, fontSize);
		TestUtilities.checkPeripheralArea("rotate", TestUtilities.BACKGROUND_COLOR, x - fontSize, y, fontSize, stringWidth, 50, 3);
	}

	/**
	 * Test drawing a string with a scale matrix.
	 *
	 * <p>
	 * The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringMatrixScale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int fontSize = 12;
		String string = STRING_SQUARE_SQUARE_SQUARE;
		int color = Colors.WHITE;
		float scaleX = 2f;
		float scaleY = 3f;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		matrix.preScale(scaleX, scaleY);

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);
		display.flush();

		// check first character
		float advance = 0;
		float firstCharacterWidth = scaleX * font.measureStringWidth(string.substring(0, 1), fontSize);
		int characterHeight = (int) (fontSize * scaleY);
		TestUtilities.checkArea("scale first character", color, (int) (x + advance), y + 1, (int) firstCharacterWidth,
				characterHeight - 2, PADDING);
		advance += firstCharacterWidth;

		// check second character
		float secondCharacterWidth = scaleX * font.measureStringWidth(string.substring(1, 2), fontSize);
		TestUtilities.checkArea("scale second character", color, (int) (x + advance), y + 1, (int) secondCharacterWidth,
				characterHeight - 2, PADDING);
		advance += secondCharacterWidth;

		// check third character
		float thirdCharacterWidth = scaleX * font.measureStringWidth(string.substring(2, 3), fontSize);
		TestUtilities.checkArea("scale third character", color, (int) (x + advance), y + 1, (int) thirdCharacterWidth,
				characterHeight - 2, PADDING);

		// Check no drawing outside
		int stringWidth = (int) (font.measureStringWidth(string, fontSize) * scaleX);
		TestUtilities.checkPeripheralArea("scale", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, characterHeight, 50, 3);
	}

	/**
	 * Test drawing a string with a scale and rotation matrix.
	 *
	 * <p>
	 * The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringMatrixRotateScale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int fontSize = 16;
		String string = STRING_SQUARE_SQUARE_SQUARE;
		int color = Colors.WHITE;
		float scaleX = 2f;
		float scaleY = 3f;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		matrix.preRotate(-90);
		matrix.preScale(scaleX, scaleY);

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);
		display.flush();

		// check first character
		float advance = 0;
		float firstCharacterWidth = scaleX * font.measureStringWidth(string.substring(0, 1), fontSize);
		int characterHeight = (int) (fontSize * scaleY);
		advance += firstCharacterWidth;
		TestUtilities.checkArea("scale rotate first character", color, x, (int) (y - advance), characterHeight,
				(int) firstCharacterWidth, PADDING);

		// check second character
		float secondCharacterWidth = scaleX * font.measureStringWidth(string.substring(1, 2), fontSize);
		advance += secondCharacterWidth;
		TestUtilities.checkArea("scale rotate second character", color, x, (int) (y - advance), characterHeight,
				(int) secondCharacterWidth, PADDING);

		// check third character
		float thirdCharacterWidth = scaleX * font.measureStringWidth(string.substring(2, 3), fontSize);
		advance += thirdCharacterWidth;
		TestUtilities.checkArea("scale rotate third character", color, x, (int) (y - advance), characterHeight,
				(int) thirdCharacterWidth, PADDING);

		// Check no drawing outside
		int stringWidth = (int) (font.measureStringWidth(string, fontSize) * scaleX);
		TestUtilities.checkPeripheralArea("scale rotate", TestUtilities.BACKGROUND_COLOR, x, (int) (y - advance), characterHeight,
				stringWidth, 50, 3);
	}

	/**
	 * Test drawing a string with a gradient.
	 *
	 * <p>
	 * The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringGradient() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int fontSize = 70;
		String string = STRING_SQUARE_BASELINE_SQUARE;
		int stringWidth = (int) font.measureStringWidth(string, fontSize);

		int x = display.getWidth() / 5;
		int y = display.getHeight() / 5;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(Colors.RED);

		// Colors must be provided with alpha channel
		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0.2f, 0.8f };

		LinearGradient gradient = new LinearGradient(0, 0, stringWidth - 1, 0, colors, stops);
		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, matrix, gradient, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);
		display.flush();

		// Read 1 pixel (avoid border of char as color can differ in EMB due to
		// antialiasing)
		TestUtilities.check("gradient start", x + 2, y, g, Colors.BLUE);
		TestUtilities.check("gradient end", x + stringWidth - 3, y, g, Colors.GREEN);

	}

	/**
	 * Test drawing a string with no corresponding glyph in the font.
	 */
	@Test
	public void testStringNoGlyph() {

		// M0092MEJAUI-2600
		// Android doesn't use the .notdef glyph provided in the font,
		// instead it renders the X character with a default system font.
		if (TestUtilities.isOn(VEEPort.Android)) {
			return;
		}

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		String string = STRING_NO_GLYPH;
		int fontSize = 100;

		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		int charWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		TestUtilities.checkArea("no glyph - left", TestUtilities.BACKGROUND_COLOR, x, y, charWidth / 2, fontSize * 80 / 100, PADDING);
		TestUtilities.checkArea("no glyph - top right", Colors.WHITE, x + charWidth / 2, y, charWidth / 2,
				fontSize * 20 / 100, PADDING);
		TestUtilities.checkArea("no glyph - bottom left", Colors.WHITE, x, y + fontSize * 80 / 100, charWidth / 2,
				fontSize * 20 / 100, PADDING);
	}

	/**
	 * Test drawing a string with clipping.
	 */
	@Test
	public void testStringClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		String string = STRING_SQUARE_BASELINE_SQUARE;
		int fontSize = 100;
		int charWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		int color = Colors.WHITE;

		g.setColor(color);
		int clipWidth = charWidth / 4;
		int clipHeight = fontSize / 2;
		g.setClip(x, y, clipWidth, clipHeight);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		g.resetClip();
		// test pixels inside the character
		TestUtilities.checkArea("clip (inside)", color, x, y, clipWidth, clipHeight);
		// test pixels outside the character
		TestUtilities.checkPeripheralArea("clip", TestUtilities.BACKGROUND_COLOR, x, y, clipWidth, clipHeight, 50, 0);

	}

	/**
	 * Test drawing a string with matrix and clipping.
	 */
	@Test
	public void testStringWithMatrixClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		String string = STRING_SQUARE_BASELINE_SQUARE;
		int fontSize = 100;
		int charWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		int color = Colors.WHITE;

		g.setColor(color);
		int clipWidth = charWidth / 3;
		int clipHeight = fontSize / 2;
		g.setClip(x, y, clipWidth, clipHeight);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		g.resetClip();
		// test pixels inside the character
		TestUtilities.checkArea("clip (inside)", color, x, y, clipWidth - 1, clipHeight - 1);
		// test pixels outside the character
		TestUtilities.checkPeripheralArea("clip", TestUtilities.BACKGROUND_COLOR, x, y, clipWidth, clipHeight, 50, 3);
	}

	/**
	 * Tests that the flush limits are correctly computed when drawing a string with gradient.
	 */
	@Test
	public void testGradientStringClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;
		int fontSize = 100;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		// gradient definition
		int stringWidth = (int) font.measureStringWidth(string, fontSize);
		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0, 1 };
		LinearGradient gradient = new LinearGradient(0, 0, stringWidth - 1, 0, colors, stops);

		int charWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		int clipWidth = charWidth / 3;
		int clipHeight = fontSize / 2;
		g.setClip(x, y, clipWidth, clipHeight);

		g.setColor(color);
		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, matrix, gradient, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER, 0);
		display.flush();

		g.resetClip();

		// Check character
		TestUtilities.checkNot("clip gradient top-left", x, y, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.checkNot("clip gradient bottom-left", x, y + clipHeight - 1, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.checkNot("clip gradient top-right", x + clipWidth - 1, y, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.checkNot("clip gradient bottom-right", x + clipWidth - 1, y + clipHeight - 1, g, TestUtilities.BACKGROUND_COLOR);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("clip gradient", TestUtilities.BACKGROUND_COLOR, x, y, clipWidth, clipHeight, 50, 3);
	}

	/**
	 * Test drawing a string with a translation vector applied to the graphics context.
	 */
	@Test
	public void testStringGCTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		String string = STRING_SQUARE;
		VectorFont font = getTestFont();
		int fontSize = 100;
		int charWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		int color = Colors.WHITE;

		int x = 100;
		int y = 100;
		int translateX = 50;
		int translateY = 50;

		g.setColor(color);
		g.translate(translateX, translateY);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		g.resetTranslation();

		// test pixels inside the character
		TestUtilities.checkArea("GC translate", color, x + translateX, y + translateY, charWidth, fontSize, PADDING);
		// test pixels outside the character
		TestUtilities.checkPeripheralArea("GC translate", TestUtilities.BACKGROUND_COLOR, x + translateX, y + translateY, charWidth,
				fontSize, 20, 3);
	}

	private static VectorFont getTestFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf");
	}

	/**
	 * Run all tests in a row with a translation vector applied on the graphics context.
	 */
	@Test
	public void testStringGCTranslated() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		// Translate the graphics context origin
		g.translate(22, 16);

		testStringAlpha();
		TestUtilities.clearScreen();

		testStringClip();
		TestUtilities.clearScreen();

		testStringColor();
		TestUtilities.clearScreen();

		testStringGradient();
		TestUtilities.clearScreen();

		testStringSize();
		TestUtilities.clearScreen();

		testStringNoGlyph();

		// Translate back gc
		g.resetTranslation();
	}

	/**
	 * Test that calling the drawString methods on a closed font throws an exception.
	 */
	@Test
	public void testStringFontIsClosed() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		float x = display.getWidth() / 2;
		float y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		VectorFont font = getTestFont();

		// close the font
		font.close();

		int errorCode = Integer.MAX_VALUE;
		try {
			VectorGraphicsPainter.drawString(g, STRING_SQUARE, font, 30, x, y);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("drawString x,y on closed font", VectorGraphicsException.RESOURCE_CLOSED, errorCode);

		errorCode = Integer.MAX_VALUE;
		try {
			VectorGraphicsPainter.drawString(g, STRING_SQUARE, font, 30, matrix, GraphicsContext.OPAQUE,
					BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("drawString x,y full on closed font", VectorGraphicsException.RESOURCE_CLOSED, errorCode);

		errorCode = Integer.MAX_VALUE;
		try {
			VectorGraphicsPainter.drawString(g, STRING_SQUARE, font, 30, matrix, GraphicsContext.OPAQUE,
					BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("drawString matrix full on closed font", errorCode,
				VectorGraphicsException.RESOURCE_CLOSED);

		int[] colors = { Colors.BLUE, Colors.GREEN };
		float[] positions = { 0x0, 0xFF };
		LinearGradient gradient = new LinearGradient(0, 0, 1, 1, colors, positions);
		errorCode = Integer.MAX_VALUE;
		try {
			VectorGraphicsPainter.drawGradientString(g, STRING_SQUARE, font, 30, matrix, gradient,
					GraphicsContext.OPAQUE, BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("drawGradientString on closed font", VectorGraphicsException.RESOURCE_CLOSED, errorCode);
	}

	/**
	 * Test drawing a string with an alpha out of range.
	 */
	@Test
	public void testStringAlphaLimits() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int fontSize = 48;
		String string = STRING_SQUARE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		int charWidth = (int) font.measureStringWidth(string, fontSize);

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, -1, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw string matrix - alpha < 0", TestUtilities.BACKGROUND_COLOR, x, y, charWidth, fontSize, PADDING);

		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, 256, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw string matrix - alpha > 255", TestUtilities.BACKGROUND_COLOR, x, y, charWidth, fontSize, PADDING);

		// Colors must be provided with alpha channel
		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0, 1 };

		LinearGradient gradient = new LinearGradient(0, 0, charWidth - 1, 0, colors, stops);

		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, matrix, gradient, -1, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw gradient string - alpha < 0", TestUtilities.BACKGROUND_COLOR, x, y, charWidth, fontSize, PADDING);

		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, matrix, gradient, 256, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw gradient string - alpha > 255", TestUtilities.BACKGROUND_COLOR, x, y, charWidth, fontSize, PADDING);
	}

	/**
	 * Test drawing a string with a size <= 0.
	 */
	@Test
	public void testStringSizeLimits() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		int color = Colors.WHITE;
		String string = STRING_SQUARE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, 0, x, y);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw string x,y - size = 0", TestUtilities.BACKGROUND_COLOR, x, y, 100, 100);

		VectorGraphicsPainter.drawString(g, string, font, -1, x, y);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw string x,y - size < 0", TestUtilities.BACKGROUND_COLOR, x, y, 100, 100);

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		VectorGraphicsPainter.drawString(g, string, font, 0, matrix, 0, BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw string matrix - size = 0", TestUtilities.BACKGROUND_COLOR, x, y, 100, 100);

		VectorGraphicsPainter.drawString(g, string, font, 0, matrix, -1, BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw string matrix - size < 0", TestUtilities.BACKGROUND_COLOR, x, y, 100, 100);

		// Colors must be provided with alpha channel
		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0, 1 };

		LinearGradient gradient = new LinearGradient(0, 0, 10, 0, colors, stops);

		VectorGraphicsPainter.drawGradientString(g, string, font, 0, matrix, gradient, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw gradient string - size = 0", TestUtilities.BACKGROUND_COLOR, x, y, 100, 100, PADDING);

		VectorGraphicsPainter.drawGradientString(g, string, font, -1, matrix, gradient, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);

		// check that nothing has been drawn
		TestUtilities.checkArea("draw gradient string - size < 0", TestUtilities.BACKGROUND_COLOR, x, y, 100, 100, PADDING);
	}

	/**
	 * Test drawing a string with a positive letter spacing.
	 */
	@Test
	public void testStringPositiveLetterSpacing() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;
		int fontSize = 48;
		int baselinePosition = (int) font.getBaselinePosition(fontSize);
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		int letterSpacing = 20;
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				letterSpacing);
		display.flush();

		// Check first character : full box filled
		int advance = 0;
		int firstCharWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		TestUtilities.checkArea("positive letter spacing first character", color, x, y, firstCharWidth, fontSize,
				PADDING);
		advance += firstCharWidth + letterSpacing;

		// Check second character: above baseline filled, nothing below
		// above baseline is color
		int secondCharWidth = (int) font.measureStringWidth(string.substring(1, 2), fontSize);
		TestUtilities.checkArea("positive letter spacing second character above baseline", color, x + advance, y,
				secondCharWidth, baselinePosition, PADDING);

		// below baseline is black
		TestUtilities.checkArea("positive letter spacing second character below baseline", TestUtilities.BACKGROUND_COLOR,
				x + advance - letterSpacing, y + baselinePosition, secondCharWidth + 2 * letterSpacing,
				fontSize - baselinePosition, PADDING);
		advance += secondCharWidth + letterSpacing;

		// Check 3rd character: full box filled
		int thirdCharWidth = (int) font.measureStringWidth(string.substring(2, 3), fontSize);
		TestUtilities.checkArea("positive letter spacing third character", color, x + advance, y, thirdCharWidth,
				fontSize, PADDING);
		int stringWidth = advance + thirdCharWidth;

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("positive letter spacing", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize, 50, 3);
	}

	/**
	 * Test drawing a string with a negative letter spacing.
	 */
	@Test
	public void testStringNegativeLetterSpacing() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;
		int fontSize = 48;
		int baselinePosition = (int) font.getBaselinePosition(fontSize);
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		int letterSpacing = -20;
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				letterSpacing);
		display.flush();

		// Check first character : full box filled
		int advance = 0;
		int firstCharWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		TestUtilities.checkArea("negative letter spacing first character", color, x, y, firstCharWidth, fontSize,
				PADDING);
		advance += firstCharWidth + letterSpacing;

		// Check second character: above baseline filled, nothing below
		// above baseline is color
		int secondCharWidth = (int) font.measureStringWidth(string.substring(1, 2), fontSize);
		TestUtilities.checkArea("negative letter spacing second character above baseline", color, x + advance, y,
				secondCharWidth, baselinePosition, PADDING);

		// below baseline is black
		TestUtilities.checkArea("negative letter spacing second character below baseline", TestUtilities.BACKGROUND_COLOR,
				x + advance - letterSpacing, y + baselinePosition, secondCharWidth + 2 * letterSpacing,
				fontSize - baselinePosition, PADDING);
		advance += secondCharWidth + letterSpacing;

		// Check 3rd character: full box filled
		int thirdCharWidth = (int) font.measureStringWidth(string.substring(2, 3), fontSize);
		TestUtilities.checkArea("negative letter spacing third character", color, x + advance, y, thirdCharWidth,
				fontSize, PADDING);
		int stringWidth = advance + thirdCharWidth;

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("negative letter spacing", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize, 50, 3);
	}

	/**
	 * Test the flush bounds set during the drawing of a string.
	 */
	@Test
	public void testStringFlushBounds() {

		if (TestUtilities.isOn(VEEPort.Android) || TestUtilities.isOn(VEEPort.JavaFX)) {
			// Flush bounds are not available on Android and JavaFX
			return;
		}

		Display display = Display.getDisplay();

		int fontSize = 48;
		int x = display.getWidth() / 2 - fontSize / 2;
		int y = display.getHeight() / 2 - fontSize / 2;

		GraphicsContext g = display.getGraphicsContext();

		// ensure flush bounds are reset
		display.flush();

		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);

		// get flush bounds
		int[] drawingLimits = new int[4];
		assertTrue(DisplayUtilities.getNextFlushBounds(drawingLimits));

		// show the results
		display.flush();

		// check the bounds (delta == 1 for the antialiasing)
		assertEquals(y, drawingLimits[1/* ymin */], 1);
		assertEquals(y + fontSize - 1, drawingLimits[3/* ymax */], 1);
	}

	/**
	 * Test that character are connected bbox to bbox.
	 */
	@Test
	public void testStringInterCharacterSpace() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_MID_SQUARE;

		int fontSize = 50;
		int x = display.getWidth() / 6;
		int y = display.getHeight() / 4;

		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		int areaHeight = fontSize / 2;
		y += font.getBaselinePosition(fontSize) - areaHeight;

		int areaWidth = fontSize;
		TestUtilities.checkArea("color character 1", Colors.WHITE, x, y, areaWidth, areaHeight, PADDING);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.1f);
		TestUtilities.checkArea("color beetween character 1 and 2 ", TestUtilities.BACKGROUND_COLOR, x, y, areaWidth, areaHeight,
				PADDING);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.25f);
		TestUtilities.checkArea("color character 2 ", Colors.WHITE, x, y, areaWidth, areaHeight, PADDING);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.65f);
		TestUtilities.checkArea("color beetween character 2 and 3 ", TestUtilities.BACKGROUND_COLOR, x, y, areaWidth, areaHeight,
				PADDING);
		x += areaWidth;

		areaWidth = (fontSize);
		TestUtilities.checkArea("color character 3 ", Colors.WHITE, x, y, areaWidth, areaHeight, PADDING);
		x += areaWidth;
	}

	/**
	 * Test that first character glyph is drawn at the origin of drawstring whether the space is in the bbox or not.
	 */
	@Test
	public void testStringFirstGlyphL() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		int fontSize = 50;

		// First char glyph is inside the bbox
		int x = display.getWidth() / 4;
		int y = display.getHeight() / 4;

		g.setColor(Colors.GREEN);
		Painter.drawLine(g, x, 0, x, display.getHeight());

		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawString(g, STRING_MID_SQUARE, font, fontSize, x, y);
		display.flush();

		int areaHeight = fontSize / 2;
		y += font.getBaselinePosition(fontSize) - areaHeight;

		int areaWidth = (int) (fontSize * 0.25f);
		TestUtilities.checkArea("color character 1", Colors.WHITE, x, y, areaWidth, areaHeight, PADDING);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.65f);
		TestUtilities.checkArea("color beetween character 1 and 2 ", TestUtilities.BACKGROUND_COLOR, x, y, areaWidth, areaHeight,
				PADDING);
		x += areaWidth;

		areaWidth = (fontSize);
		TestUtilities.checkArea("color character 2 ", Colors.WHITE, x, y, areaWidth, areaHeight, PADDING);
		x += areaWidth;
	}

	/**
	 * Test that first character glyph is drawn at the origin of drawstring whatever the space is in the bbox or not.
	 */
	@Test
	public void testStringFirstGlyphM() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		int fontSize = 50;

		// First char glyph is inside the bbox
		int x = display.getWidth() / 4;
		int y = display.getHeight() / 4;

		g.setColor(Colors.GREEN);
		Painter.drawLine(g, x, 0, x, display.getHeight());

		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawString(g, STRING_OUTLEFT_SQUARE, font, fontSize, x, y);
		display.flush();

		int areaHeight = fontSize / 3;
		y += font.getBaselinePosition(fontSize) - areaHeight;

		int areaWidth = (int) (fontSize * 0.9f);
		TestUtilities.checkArea("color character 1", Colors.WHITE, x, y, areaWidth, areaHeight, PADDING);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.4f);
		TestUtilities.checkArea("color beetween character 1 and 2 ", TestUtilities.BACKGROUND_COLOR, x, y, areaWidth, areaHeight,
				PADDING);
		x += areaWidth;

		areaWidth = (fontSize);
		TestUtilities.checkArea("color character 2 ", Colors.WHITE, x, y, areaWidth, areaHeight, PADDING);
		x += areaWidth;
	}

	/**
	 * Test that first character glyph is draw from first inflexion point of first glyph
	 */

	@Test
	public void testStringFirstGlyphLeftInflexionPoint() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		int fontSize = 100;

		// First char glyph is inside the bbox
		int x = display.getWidth() / 4;
		int y = display.getHeight() / 4;

		g.setColor(Colors.GREEN);
		Painter.drawLine(g, x, 0, x, display.getHeight());

		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawString(g, STRING_LEFTINFLEXIONPOINT, font, fontSize, x, y);
		display.flush();

		y += font.getBaselinePosition(fontSize) - 0.06 * fontSize;

		int areaWidth = 1;
		TestUtilities.checkArea("green line color ", Colors.GREEN, x, y, areaWidth, 1, 0);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.04f);
		TestUtilities.checkArea("before character line color ", TestUtilities.BACKGROUND_COLOR, x, y, areaWidth, 1, 0);
		areaWidth += (int) (fontSize * 0.02f);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.28f);
		TestUtilities.checkArea("in character", Colors.WHITE, x, y, areaWidth, 1, 0);
		areaWidth += (int) (fontSize * 0.03f);
		x += areaWidth;

		areaWidth = (int) (fontSize * 0.10f);
		TestUtilities.checkArea("after character", TestUtilities.BACKGROUND_COLOR, x, y, areaWidth, 1, 0);
		x += areaWidth;
	}

	/**
	 * Test that first character glyph is draw from first inflexion point of first glyph
	 */
	@Test
	public void testStringWithSpace() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		int fontSize = 100;

		// First char glyph is inside the bbox
		int x = display.getWidth() / 4;
		int y = display.getHeight() / 4;

		g.setColor(Colors.GREEN);
		Painter.drawLine(g, x - 1, 0, x - 1, display.getHeight());

		g.setColor(Colors.WHITE);

		// space String
		VectorGraphicsPainter.drawString(g, STRING_SPACE, font, fontSize, x, y);
		TestUtilities.checkArea("space character", TestUtilities.BACKGROUND_COLOR, x, y, fontSize, fontSize, PADDING);

		// space at start of string
		VectorGraphicsPainter.drawString(g, STRING_LEFTSPACE, font, fontSize, x, y);
		TestUtilities.checkArea("space at start of string", Colors.WHITE, x, y,
				(int) font.measureStringWidth(STRING_SQUARE, fontSize), fontSize, PADDING);

		// space at end of string
		y += 150;
		VectorGraphicsPainter.drawString(g, STRING_RIGHTSPACE, font, fontSize, x, y);
		TestUtilities.checkArea("space at start of string", Colors.WHITE, x, y,
				(int) font.measureStringWidth(STRING_SQUARE, fontSize), fontSize, PADDING);

		// space inside the string
		y += 150;
		VectorGraphicsPainter.drawString(g, STRING_SPACEINSIDE, font, fontSize, x, y);
		TestUtilities.checkArea("space inside the string", Colors.WHITE, x, y,
				2 * (int) font.measureStringWidth(STRING_SQUARE, fontSize), fontSize, PADDING);

		display.flush();
	}

	/**
	 * Test that second string is drawn when the first string is an "empty" drawing
	 */
	@Test
	public void testEmptyString() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		int fontSize = 48;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		int firstCharWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);

		g.setColor(Colors.WHITE);

		// Draw an empty drawing: must work and not lock the drawing engine.
		VectorGraphicsPainter.drawString(g, "", font, fontSize, x, y);
		TestUtilities.checkArea("Empty string (standard)", TestUtilities.BACKGROUND_COLOR, x, y, firstCharWidth, fontSize, PADDING);

		// Nothing should be drawn either with a gradient string.
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		LinearGradient gradient = new LinearGradient(0, 0, display.getWidth(), display.getHeight(),
				new int[] { Colors.RED, Colors.BLUE });
		VectorGraphicsPainter.drawGradientString(g, "", font, fontSize, new Matrix(), gradient, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);
		TestUtilities.checkArea("Empty string (gradient)", TestUtilities.BACKGROUND_COLOR, x, y, firstCharWidth, fontSize, PADDING);

		// draw a character as usual
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		// Check character : full box filled
		TestUtilities.checkArea("color character", Colors.WHITE, x, y, firstCharWidth, fontSize, PADDING);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("color", TestUtilities.BACKGROUND_COLOR, x, y, firstCharWidth, fontSize, 50, 3);
	}

	/**
	 * Test that second string is drawn when the first string is drawn outside the clip (== empty drawing)
	 */
	@Test
	public void testStringWithClip() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		int fontSize = 48;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		int firstCharWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);

		g.setColor(Colors.WHITE);

		g.setClip(0, 0, 10, 10);

		// Draw a string outside the clip: nothing is drawn.
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		TestUtilities.checkArea("String outside of clip (standard)", TestUtilities.BACKGROUND_COLOR, x, y, firstCharWidth, fontSize,
				PADDING);

		// Nothing should be drawn either with a gradient string.
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		LinearGradient gradient = new LinearGradient(0, 0, display.getWidth(), display.getHeight(),
				new int[] { Colors.RED, Colors.BLUE });
		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, new Matrix(), gradient,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);
		TestUtilities.checkArea("String outside of clip (gradient)", TestUtilities.BACKGROUND_COLOR, x, y, firstCharWidth, fontSize,
				PADDING);

		g.resetClip();

		// draw a character as usual
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		// Check character : full box filled
		TestUtilities.checkArea("color character", Colors.WHITE, x, y, firstCharWidth, fontSize, PADDING);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("color", TestUtilities.BACKGROUND_COLOR, x, y, firstCharWidth, fontSize, 50, 3);
	}

	/**
	 * Tests the drawing of all kinds of characters (with or without oversizing before and after the left and right
	 * limits).
	 */
	@Test
	public void testMeasureSingleCharacter() {
		testCharacter(FirstFont.CENTERED);
		testCharacter(FirstFont.LEFTLIMIT);
		testCharacter(FirstFont.LEFTOVER);
		testCharacter(FirstFont.RIGHTLIMIT);
		testCharacter(FirstFont.RIGHTOVER);
		testCharacter(FirstFont.LEFTLIMITRIGHTOVER);
		testCharacter(FirstFont.LEFTOVERRIGHTLIMIT);
		testCharacter(FirstFont.OVER);

		testCharacter(FirstFont.OUTSIDELEFTLIMIT);
		testCharacter(FirstFont.OUTSIDELEFTOVER);
		testCharacter(FirstFont.OUTSIDERIGHTLIMIT);
		testCharacter(FirstFont.OUTSIDERIGHTOVER);
		testCharacter(FirstFont.NULLLEFTLIMIT);
		testCharacter(FirstFont.NULLLEFTOVER);
		testCharacter(FirstFont.NULLRIGHTLIMIT);
		testCharacter(FirstFont.NULLRIGHTOVER);
	}

	private void testCharacter(FirstFontCharacter character) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		TestUtilities.clearScreen();
		VectorFont font = getTestFont();

		int scale = 10;
		int fontSize = FirstFont.EM_SIZE / scale;
		int x = 100;
		int y = 100;
		int color = Colors.WHITE;
		int foregroundColor = display.getDisplayColor(color);
		int backgroundColor = display.getDisplayColor(TestUtilities.BACKGROUND_COLOR);
		int baseline = (int) font.getBaselinePosition(fontSize);

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, character.string(), font, fontSize, x, y);
		display.flush();

		int padding = 3;
		y += baseline - 20;

		Assert.assertTrue(character.string() + " left outside",
				TestUtilities.compare(backgroundColor, g.readPixel(x - padding, y)));
		Assert.assertTrue(character.string() + " left inside",
				TestUtilities.compare(foregroundColor, g.readPixel(x + padding, y)));
		Assert.assertTrue(character.string() + " right inside",
				TestUtilities.compare(foregroundColor, g.readPixel(x + character.glyphWidth() / scale - padding, y)));
		Assert.assertTrue(character.string() + " right outside",
				TestUtilities.compare(backgroundColor, g.readPixel(x + character.glyphWidth() / scale + padding, y)));
	}

	/**
	 * Test the drawing of all combinations of all character types (with or without oversizing before and after the left
	 * and right limits).
	 */
	@Test
	public void testMeasureTwoCharacters() {
		FirstFontCharacter[] characters = new FirstFontCharacter[] { FirstFont.CENTERED, FirstFont.LEFTLIMIT,
				FirstFont.LEFTOVER, FirstFont.RIGHTLIMIT, FirstFont.RIGHTOVER, FirstFont.LEFTLIMITRIGHTOVER,
				FirstFont.LEFTOVERRIGHTLIMIT, FirstFont.OVER, FirstFont.OUTSIDELEFTLIMIT, FirstFont.OUTSIDELEFTOVER,
				FirstFont.OUTSIDERIGHTLIMIT, FirstFont.OUTSIDERIGHTOVER, FirstFont.NULLLEFTLIMIT,
				FirstFont.NULLLEFTOVER, FirstFont.NULLRIGHTLIMIT, FirstFont.NULLRIGHTOVER };

		for (FirstFontCharacter a : characters) {
			for (FirstFontCharacter b : characters) {
				testTwoCharacters(a, b);
			}
		}
	}

	private void testTwoCharacters(FirstFontCharacter a, FirstFontCharacter b) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		TestUtilities.clearScreen();
		VectorFont font = getTestFont();
		String s = a.string() + b.string();

		int scale = 10;
		int fontSize = FirstFont.EM_SIZE / scale;
		int x = 100;
		int y = 100;
		int color = Colors.WHITE;
		int foregroundColor = display.getDisplayColor(color);
		int backgroundColor = display.getDisplayColor(TestUtilities.BACKGROUND_COLOR);
		int baseline = (int) font.getBaselinePosition(fontSize);

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, s, font, fontSize, x, y);
		display.flush();

		int padding = 3;
		y += baseline - 20;

		// wheelbase of first character: remove left space except if bx is negative:
		// so width = ax - ls +
		// * bx > 0 -> bx == ls -> -bx + ls = 0
		// * bx < 0 -> -bx == abs(bx)
		int aWidth = a.advanceX() /*- a.leftSpace()*/ - a.bearingX() /* + a.leftSpace() */;

		// wheelbase of second character: remove right space and bx when is negative ("b" overlaps "a"):
		// so width = ls + gw +
		// * bx > 0 -> bx == ls -> bx - ls = 0
		// * bx < 0 -> ls == 0 -> bx == -abs(bx)
		int bWidth = /* b.leftSpace() + */ b.glyphWidth() + b.bearingX() /*- b.leftSpace()*/;

		int aStart = x;
		int aEnd = aStart + a.glyphWidth() / scale - 1;
		int bStart = aStart + (aWidth /* + b.leftSpace() */ + b.bearingX() /*- b.leftSpace()*/) / scale;
		int bEnd = bStart + b.glyphWidth() / scale - 1;

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			// see also TestFontMeasures.testMeasureTwoCharacters()
			System.out.println(s + " : " + aStart + " to " + aEnd + " and " + bStart + " to " + bEnd);
			System.out.println(s + " : " + aWidth + " + " + bWidth);
		}

		if (aEnd + padding < bStart - padding) {
			// (a) before (b) and space between
			testTwoCharacters(g, s, x, y, foregroundColor, backgroundColor, backgroundColor, aStart, aEnd, bStart, bEnd,
					padding);

		} else if (bEnd + padding < aStart - padding) {
			// (b) before (a) and space between
			testTwoCharacters(g, s, x, y, foregroundColor, backgroundColor, backgroundColor, bStart, bEnd, aStart, aEnd,
					padding);

		} else {

			// (a) overlaps (b) or vice-versa

			int left = Math.min(aStart, bStart);
			int right = Math.max(aEnd, bEnd);

			// easier to test two times the same point in the middle
			int middleMin = left == aStart ? bStart : aStart;
			int middleMax = middleMin == aEnd ? bEnd : aEnd;
			int middle = middleMin + (middleMax - middleMin) / 2;

			testTwoCharacters(g, s, x, y, foregroundColor, backgroundColor, foregroundColor, left, middle, middle,
					right, padding);
		}
	}

	private void testTwoCharacters(GraphicsContext g, String s, int x, int y, int foregroundColor, int backgroundColor,
			int middleColor, int aStart, int aEnd, int bStart, int bEnd, int padding) {

		Assert.assertTrue(s + " (LeftChar) left outside",
				TestUtilities.compare(backgroundColor, g.readPixel(aStart - padding, y)));
		Assert.assertTrue(s + " (LeftChar) left inside",
				TestUtilities.compare(foregroundColor, g.readPixel(aStart + padding, y)));

		Assert.assertTrue(s + " (LeftChar) right inside",
				TestUtilities.compare(foregroundColor, g.readPixel(aEnd - padding, y)));
		Assert.assertTrue(s + " (LeftChar) right outside",
				TestUtilities.compare(middleColor, g.readPixel(aEnd + padding, y)));

		Assert.assertTrue(s + " (RightChar) left outside",
				TestUtilities.compare(middleColor, g.readPixel(bStart - padding, y)));
		Assert.assertTrue(s + " (RightChar) left inside",
				TestUtilities.compare(foregroundColor, g.readPixel(bStart + padding, y)));

		Assert.assertTrue(s + " (RightChar) left inside",
				TestUtilities.compare(foregroundColor, g.readPixel(bEnd - padding, y)));
		Assert.assertTrue(s + " (RightChar) left outside",
				TestUtilities.compare(backgroundColor, g.readPixel(bEnd + padding, y)));
	}
}
