/*
 * Copyright 2010-2015 http://beders.github.com/Resty
 * Copyright 2015-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ej.rest.web;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URLConnection;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

/**
 * Abstract base class for all resource handlers you want to use with Resty.
 * 
 * It gives access to the underlying URLConnection and the current inputStream
 * 
 * @author beders
 * 
 */
public abstract class AbstractResource extends Resty {
	static final Logger log = Logger.getLogger(AbstractResource.class.getName());
	protected URLConnection urlConnection;
	protected InputStream inputStream;

	public AbstractResource(Option... options) {
		super(options);
	}

	protected abstract String getAcceptedTypes();

	void fill(URLConnection anUrlConnection) throws IOException {
		urlConnection = anUrlConnection;
		try {
			inputStream = anUrlConnection.getInputStream();
		} catch (IOException e) {
			if (anUrlConnection instanceof HttpURLConnection) {
				HttpURLConnection conn = (HttpURLConnection) anUrlConnection;
				//Check if there is an error stream
				//If there is one, this mean that the server returns an http error code
				//and in this case don't throw an IOException
				InputStream stream = conn.getErrorStream();
				if (stream != null) {
					InputStream es = new BufferedInputStream(stream);
					ByteArrayOutputStream baos = new ByteArrayOutputStream();
					try {
						// read the response body
						byte[] buf = new byte[1024];
						int read = -1;
						while ((read = es.read(buf)) > 0) {
							baos.write(buf, 0, read);
						}
					} catch (IOException e1) {
						log.warning("IOException when reading the error stream. Ignored");
					}

					// close the errorstream
					es.close();
					inputStream = new ByteArrayInputStream(baos.toByteArray());
				} else {
					// we close the underlying connection in embedded mode.
					conn.disconnect();
					throw e;
				}
			} else {
				throw e;
			}
		}
	}

	public URLConnection getUrlConnection() {
		return urlConnection;
	}

	public HttpURLConnection http() {
		return (HttpURLConnection) urlConnection;
	}

	public InputStream stream() {
		return inputStream;
	}

	/**
	 * Check if the URLConnection has returned the specified responseCode
	 * 
	 * @param responseCode
	 * @return
	 */
	public boolean status(int responseCode) {
		if (urlConnection instanceof HttpURLConnection) {
			HttpURLConnection http = (HttpURLConnection) urlConnection;
			try {
				return http.getResponseCode() == responseCode;
			} catch (IOException e) {
				e.printStackTrace();
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * Get the location header as URI. Returns null if there is no location header.
	 * 
	 */
	public URI location() {
		String loc = http().getHeaderField("Location");
		if (loc != null) {
			return URI.create(loc);
		}
		return null;
	}

	/** Print out the response headers for this resource.
	 * 
	 * @return
	 */
	public String printResponseHeaders() {
		StringBuilder sb = new StringBuilder();
		HttpURLConnection http = http();
		if (http != null) {
			Map<String, List<String>> header = http.getHeaderFields();
			for (String key : header.keySet()) {
				for (String val : header.get(key)) {
					sb.append(key).append(": ").append(val).append("\n");
				}
			}
		}
		return sb.toString();
	}
}
