/*
 * Java
 *
 * Copyright 2017-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net.util.wifi;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.bon.Util;
import ej.ecom.network.IPConfiguration;
import ej.ecom.network.NetworkInterfaceManager;
import ej.ecom.wifi.AccessPoint;
import ej.ecom.wifi.SecurityMode;
import ej.ecom.wifi.SoftAPConfiguration;
import ej.ecom.wifi.WifiCapability;
import ej.ecom.wifi.WifiManager;
import ej.net.util.NetUtil;
import ej.util.message.Level;

/**
 * This manager manage the Wi-Fi manager and network to keep a state.
 *
 * @see WifiManager
 * @see NetworkInterfaceManager
 */
public class WifiNetworkManager {

	private static final int WIFI_IP_ASSIGNED_POLLING_DELAY_DEFAULT_MS = 1000;
	private static int wifiIPAcquirePollingDelayMs = Integer
			.getInteger("ej.net.util.wifi.ip.assigned.polling.delay.ms", WIFI_IP_ASSIGNED_POLLING_DELAY_DEFAULT_MS)
			.intValue();

	private static final String INTERFACE_NAME_PROPERTY = "wifi.interface.name"; //$NON-NLS-1$

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$

	private final WifiManager wifiManager;

	private NetworkInterface wifiInterface;

	private AccessPointConfiguration configurationAP;
	private SoftAPConfiguration softAPConfiguration;
	private boolean scanWhileSoftAP;
	private boolean isSoftAP;
	private IPConfiguration clientIPConfiguration;
	private IPConfiguration softAPIPConfiguration;

	private String interfaceName = System.getProperty(INTERFACE_NAME_PROPERTY);

	/**
	 * Instantiates a WifiNetworkManager with a default {@link AccessPointConfiguration} and
	 * {@link SoftAPConfiguration}.
	 *
	 * @throws IOException
	 *             When initialise fail.
	 * @see #WifiNetworkManager(AccessPointConfiguration, SoftAPConfiguration)
	 */
	public WifiNetworkManager() throws IOException {
		this(new AccessPointConfiguration(EMPTY_STRING), new SoftAPConfiguration());
	}

	/**
	 * Instantiates a WifiNetworkManager.
	 *
	 * @param configurationAP
	 *            the configurationAP to use.
	 * @param softAPConfiguration
	 *            the softAPConfiguration to use.
	 * @throws IOException
	 *             When initialise fail.
	 * @throws NullPointerException
	 *             if parameters are <code>null</code> or {@link WifiManager} not found.
	 * @see WifiManager#getInstance()
	 */
	public WifiNetworkManager(@NonNull AccessPointConfiguration configurationAP,
			@NonNull SoftAPConfiguration softAPConfiguration) throws IOException, NullPointerException {
		if (configurationAP == null || softAPConfiguration == null) {
			throw new NullPointerException();
		}
		this.configurationAP = configurationAP;
		this.softAPConfiguration = softAPConfiguration;
		this.wifiManager = WifiManager.getInstance();
		if (this.wifiManager == null) {
			throw new NullPointerException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.WIFI_CATEGORY,
					Messages.ERROR_NO_WIFI_MANAGER));
		}
		this.wifiInterface = null;
		this.scanWhileSoftAP = true;
		this.isSoftAP = false;
		this.clientIPConfiguration = getDefaultIPConfiguration();
		initNetwork();
	}

	/**
	 * Checks if the Wi-FI Manager is initialized.
	 *
	 * @return <code>true</code> if the Wi-FI manager is initialized.
	 * @see NetworkInterfaceManager#isEnabled(NetworkInterface)
	 */
	public boolean isInit() {
		return this.wifiInterface != null // && NetworkInterfaceManager.isStarted(this.wifiInterface)
				&& NetworkInterfaceManager.isEnabled(this.wifiInterface);
	}

	/**
	 * Scans Access Points.
	 *
	 * @return the list of available access points (it is empty if there is no available access point).
	 * @throws IOException
	 *             if an error occurred.
	 * @see WifiManager#scan()
	 */
	public AccessPoint[] scanAccessPoints() throws IOException {
		boolean wasMounted = false;
		if (!this.scanWhileSoftAP && this.isSoftAP) {
			wasMounted = true;
			unmountSoftAccessPoint();
		}
		AccessPoint[] scan;
		try {
			scan = this.wifiManager.scan(false);
		} catch (IOException e) {
			if (this.scanWhileSoftAP && this.isSoftAP) {
				// Try again unmounting softap first.
				setScanWhileSoftAP(false);
				scan = scanAccessPoints();
				wasMounted = false;
			} else {
				throw new IOException(e);
			}
		}
		if (wasMounted) {
			mountSoftAccessPoint();
		}
		return scan;
	}

	/**
	 * Joins the access point configuration selected.
	 *
	 * @param timeout
	 *            the timeout to wait for an IP, 0 for unlimited, cannot be less than 0.
	 * @throws IOException
	 *             if an error occurred.
	 * @throws NullPointerException
	 *             if AP is not set.
	 * @throws UnsupportedOperationException
	 *             If the Wi-Fi does not support client mode.
	 * @throws IllegalArgumentException
	 *             if timeout is less than 0.
	 * @see WifiNetworkManager#getAPConfiguration()
	 * @see AccessPointConfiguration#setAccessPoint(AccessPoint)
	 * @see WifiNetworkManager#joinAccessPoint(AccessPoint, String, int)
	 * @see #joinAccessPoint(AccessPoint, String, int)
	 * @see #joinAccessPoint(String, String, SecurityMode, int)
	 */
	public void joinAccessPoint(int timeout)
			throws IOException, NullPointerException, UnsupportedOperationException, IllegalArgumentException {
		AccessPoint accessPoint = this.configurationAP.getAccessPoint();
		if (accessPoint != null) {
			joinAccessPoint(accessPoint, this.configurationAP.getPassphrase(), timeout);
		} else {
			joinAccessPoint(this.configurationAP.getSSID(), this.configurationAP.getPassphrase(),
					this.configurationAP.getSecurityMode(), timeout);
		}
	}

	/**
	 * Joins a specific {@link AccessPoint}.
	 *
	 * @param access
	 *            the {@link AccessPoint} to join.
	 * @param password
	 *            the {@link AccessPoint} password.
	 * @param timeout
	 *            the timeout to wait for an IP, 0 for unlimited, cannot be less than 0.
	 * @throws IOException
	 *             if an error occurred.
	 * @throws NullPointerException
	 *             if the AP is <code>null</code>.
	 * @throws UnsupportedOperationException
	 *             If the Wi-Fi does not support client mode.
	 * @throws IllegalArgumentException
	 *             if timeout is less than 0.
	 *
	 * @see WifiManager#join(AccessPoint, String)
	 */
	public void joinAccessPoint(@NonNull AccessPoint access, @Nullable String password, int timeout)
			throws IOException, NullPointerException, UnsupportedOperationException, IllegalArgumentException {
		if (this.wifiManager.getCapability() == WifiCapability.SOFT_AP) {
			throw new UnsupportedOperationException();
		}
		if (access == null) {
			throw new NullPointerException();
		}
		if (timeout < 0) {
			throw new IllegalArgumentException();
		}

		if (this.isSoftAP && this.wifiManager.getCapability() != WifiCapability.BOTH_SIMULTANEOUS) {
			try {
				unmountSoftAccessPoint();
			} catch (IOException e) {
				Messages.LOGGER.log(Level.INFO, Messages.WIFI_CATEGORY, Messages.ERROR_DURING_UNMOUNT, e);
			}
		}
		AccessPoint joinedAP = getJoinedAccessPoint();
		if (!access.equals(joinedAP)) {
			if (joinedAP != null) {
				try {
					leaveAccessPoint();
				} catch (IOException e) {
					Messages.LOGGER.log(Level.INFO, Messages.WIFI_CATEGORY, Messages.ERROR_DURING_LEAVE, e);
				}
			}
			setCurrentIPConfiguration(this.clientIPConfiguration);

			this.wifiManager.join(access, password);
		}

		checkJoinedAP(timeout);
	}

	/**
	 * Joins a particular Access Point.
	 *
	 * @param ssid
	 *            the AP SSID.
	 * @param password
	 *            the AP password.
	 * @param securityMode
	 *            the security Mode to use.
	 * @param timeout
	 *            the timeout to wait for an IP, 0 for unlimited, cannot be less than 0.
	 *
	 * @throws IOException
	 *             if an error occurred.
	 * @throws UnsupportedOperationException
	 *             If the Wi-Fi does not support client mode.
	 * @throws IllegalArgumentException
	 *             if timeout is less than 0.
	 *
	 * @see WifiManager#join(String, String)
	 */
	public void joinAccessPoint(@NonNull String ssid, @Nullable String password, @Nullable SecurityMode securityMode,
			int timeout) throws IOException, UnsupportedOperationException, IllegalArgumentException {
		if (this.wifiManager.getCapability() == WifiCapability.SOFT_AP) {
			throw new UnsupportedOperationException();
		}
		if (this.isSoftAP && this.wifiManager.getCapability() != WifiCapability.BOTH_SIMULTANEOUS) {
			try {
				unmountSoftAccessPoint();
			} catch (IOException e) {
				Messages.LOGGER.log(Level.INFO, Messages.WIFI_CATEGORY, Messages.ERROR_DURING_UNMOUNT, e);
			}
		}
		AccessPoint joinedAP = getJoinedAccessPoint();
		if (joinedAP == null || !ssid.equals(joinedAP.getSSID())) {
			if (joinedAP != null) {
				try {
					leaveAccessPoint();
				} catch (IOException e) {
					Messages.LOGGER.log(Level.INFO, Messages.WIFI_CATEGORY, Messages.ERROR_DURING_LEAVE, e);
				}
			}
			setCurrentIPConfiguration(this.clientIPConfiguration);

			/**
			 * Joins the configured Access Point
			 */
			password = sanitizePassword(password);
			if (securityMode == null) {
				this.wifiManager.join(ssid, password);
			} else {
				this.wifiManager.join(ssid, password, securityMode);
			}
		}

		checkJoinedAP(timeout);
	}

	/**
	 * Leaves the joined Access Point.
	 *
	 * @throws IOException
	 *             if an error occurred.
	 * @see WifiManager#leave()
	 */
	public void leaveAccessPoint() throws IOException {
		AccessPoint joinedAP;

		// Check if the AP is actually joined.
		joinedAP = getJoinedAccessPoint();
		if (joinedAP == null) {
			throw new IOException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.WIFI_CATEGORY,
					Messages.ERROR_COULD_NOT_JOIN_AP));
		}

		// Leave the AP.
		this.wifiManager.leave();
		this.configurationAP.setAccessPoint(null);
	}

	/**
	 * Checks if the Wi-FI is connected to network.
	 *
	 * @return true if the Wi-FI manager is connected to a network.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @see WifiManager#getJoined()
	 */
	public boolean isConnected() throws IOException {
		return getJoinedAccessPoint() != null;
	}

	/**
	 * Checks if the Wi-FI is has enabled the softAP.
	 *
	 * @return true if enabled, false otherwise.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @see WifiManager#isSoftAPEnabled()
	 */
	public boolean isSoftAPEnabled() throws IOException {
		this.isSoftAP = this.wifiManager.isSoftAPEnabled();
		return this.isSoftAP;
	}

	/**
	 * Gets the joined access.
	 *
	 * @return the joined access, <code>null</code> if none.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @see WifiManager#getJoined()
	 */
	public AccessPoint getJoinedAccessPoint() throws IOException {
		AccessPoint joined = null;
		if (!this.isSoftAP || this.wifiManager.getCapability() == WifiCapability.BOTH_SIMULTANEOUS) {
			joined = this.wifiManager.getJoined();
		}
		return joined;
	}

	/**
	 * Gets the configurationAP.
	 *
	 * @return the configurationAP.
	 */
	public AccessPointConfiguration getAPConfiguration() {
		return this.configurationAP;
	}

	/**
	 * Sets the configurationAP.
	 *
	 * @param configurationAP
	 *            the configurationAP to set.
	 */
	public void setAPConfiguration(@NonNull AccessPointConfiguration configurationAP) {
		if (configurationAP == null) {
			throw new NullPointerException();
		}
		this.configurationAP = configurationAP;
	}

	/**
	 * Gets the softAPConfiguration.
	 *
	 * @return the softAPConfiguration.
	 */
	public SoftAPConfiguration getSoftAPConfiguration() {
		return this.softAPConfiguration;
	}

	/**
	 * Sets the softAPConfiguration.
	 *
	 * @param softAPConfiguration
	 *            the softAPConfiguration to set, cannot be <code>null</code>.
	 * @throws NullPointerException
	 *             if {@link SoftAPConfiguration} is <code>null</code>.
	 */
	public void setSoftAPConfiguration(@NonNull SoftAPConfiguration softAPConfiguration) throws NullPointerException {
		if (softAPConfiguration == null) {
			throw new NullPointerException();
		}
		this.softAPConfiguration = softAPConfiguration;
	}

	/**
	 * Mounts a Soft Access Point.
	 *
	 * @throws IOException
	 *             if an error occurred.
	 * @see WifiManager#enableSoftAP(SoftAPConfiguration)
	 */
	public void mountSoftAccessPoint() throws IOException {
		mountSoftAccessPoint(this.softAPConfiguration);
	}

	/**
	 * Mounts a Soft Access Point.
	 *
	 * @param config
	 *            the configuration to use.
	 *
	 * @throws IOException
	 *             if an error occurred.
	 *
	 * @throws UnsupportedOperationException
	 *             If the Wi-Fi does not support SoftAP mode.
	 *
	 * @see WifiManager#enableSoftAP(SoftAPConfiguration)
	 */
	public void mountSoftAccessPoint(@NonNull SoftAPConfiguration config)
			throws IOException, UnsupportedOperationException {
		setSoftAPConfiguration(config);
		if (this.wifiManager.getCapability() == WifiCapability.CLIENT) {
			throw new UnsupportedOperationException();
		}
		if (isConnected() && this.wifiManager.getCapability() != WifiCapability.BOTH_SIMULTANEOUS) {
			try {
				leaveAccessPoint();
			} catch (IOException e) {
				Messages.LOGGER.log(Level.INFO, Messages.WIFI_CATEGORY, Messages.ERROR_DURING_LEAVE, e);
			}
		}

		if (config.getSSID() == null) {
			throw new IllegalArgumentException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.WIFI_CATEGORY, Messages.ERROR_SSID_NULL));
		}

		setCurrentIPConfiguration(this.softAPIPConfiguration);
		if (config.getName() == null) {
			config.setName(config.getSSID());
		}
		/**
		 * Enable the Soft AP
		 */
		this.wifiManager.enableSoftAP(config);
		this.isSoftAP = true;
	}

	/**
	 * Unmounts a Soft Access Point.
	 *
	 * @throws IOException
	 *             if an error occurred.
	 * @see WifiManager#disableSoftAP()
	 */
	public void unmountSoftAccessPoint() throws IOException {
		if (this.isSoftAP) {
			/**
			 * Disable the Soft AP
			 */
			this.wifiManager.disableSoftAP();
			this.isSoftAP = false;
		}
	}

	/**
	 * Gets whether it supports scan while in softAP.
	 *
	 * @return <code>true</code> if scan while mounted is supported.
	 */
	public boolean supportScanWhileSoftAP() {
		return this.scanWhileSoftAP;
	}

	/**
	 * Sets whether it supports scan while mounted.
	 *
	 * @param scanWhileSoftAP
	 *            <code>true</code> if scan while mounted is supported.
	 */
	public void setScanWhileSoftAP(boolean scanWhileSoftAP) {
		this.scanWhileSoftAP = scanWhileSoftAP;
	}

	/**
	 * Gets the {@link WifiCapability}.
	 *
	 * @return the capabilities.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @see WifiManager#getCapability()
	 */
	public WifiCapability getCapabilities() throws IOException {
		return this.wifiManager.getCapability();
	}

	/**
	 * Sets the {@link IPConfiguration} for the interface while in client mode. This will be taken in account on the
	 * next join.
	 *
	 * @param ipConfiguration
	 *            the {@link IPConfiguration} to use, can be <code>null</code> then it won't be used.
	 */
	public void setClientIPConfigure(@Nullable IPConfiguration ipConfiguration) {
		this.clientIPConfiguration = ipConfiguration;
	}

	/**
	 * Sets the {@link IPConfiguration} for the interface while in softAP. This will be taken in account on the next
	 * {@link #mountSoftAccessPoint()}.
	 *
	 * @param softAPIPConfiguration
	 *            the {@link IPConfiguration} to use when in softAP.
	 */
	public void setSoftAPIPConfigure(@Nullable IPConfiguration softAPIPConfiguration) {
		this.softAPIPConfiguration = softAPIPConfiguration;
	}

	/**
	 * Gets the interfaceName.
	 *
	 * @return the interfaceName.
	 */
	public String getInterfaceName() {
		return this.interfaceName;
	}

	/**
	 * Sets the interfaceName.
	 *
	 * @param interfaceName
	 *            the interfaceName to set.
	 */
	public void setInterfaceName(@Nullable String interfaceName) {
		this.interfaceName = interfaceName;
	}

	/**
	 * Gets the Wi-FI manager used.
	 *
	 * @return the Wi-FI manager.
	 */
	public WifiManager getWifiManager() {
		return this.wifiManager;
	}

	/**
	 * Gets the wifi Interface.
	 *
	 * @return the wifiInterface.
	 */
	public NetworkInterface getWifiInterface() {
		if (this.wifiInterface == null) {
			try {
				updateInterface();
			} catch (IOException e) {
				// Not used.
			}
		}
		return this.wifiInterface;
	}

	private void setCurrentIPConfiguration(IPConfiguration ipConfiguration) throws IOException {
		NetworkInterface networkInterface = this.wifiInterface;
		if (networkInterface != null && ipConfiguration != null) {
			IPConfiguration oldConfiguration = NetworkInterfaceManager.getIPConfiguration(networkInterface);
			if (!ipConfiguration.equals(oldConfiguration)) {
				NetworkInterfaceManager.configure(networkInterface, ipConfiguration);
			}
		}
	}

	private void checkJoinedAP(int timeout) throws IOException {
		AccessPoint joinedAP;
		waitUntilIpIsAssigned(timeout);

		/**
		 * Check if the AP is actually joined
		 */
		joinedAP = getJoinedAccessPoint();
		if (joinedAP != null) {
			this.configurationAP.setAccessPoint(joinedAP);
		} else {
			throw new IOException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.WIFI_CATEGORY,
					Messages.ERROR_COULD_NOT_JOIN_AP));
		}
	}

	private String sanitizePassword(String password) {
		return (password == null) ? EMPTY_STRING : password;
	}

	/**
	 * Update the interface.
	 *
	 * @throws IOException
	 *             if an IOException occurs.
	 */
	private void updateInterface() throws IOException {
		NetworkInterface wifiInterface = this.wifiInterface;
		String name = this.interfaceName;
		if (name != null) {
			wifiInterface = NetUtil.getInterface(name);
		} else {
			wifiInterface = getDefaultInterface();
		}
		this.wifiInterface = wifiInterface;
	}

	/**
	 * @return
	 * @throws SocketException
	 */
	private NetworkInterface getDefaultInterface() throws SocketException {
		Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
		NetworkInterface networkInterface = null;
		/**
		 * Look through all of the enumeration and compare with the one to be found
		 */
		while (interfaces.hasMoreElements()) {
			NetworkInterface currentInterface = interfaces.nextElement();
			if (!currentInterface.isLoopback()) {
				if (currentInterface.isUp()) {
					Enumeration<InetAddress> inetAddresses = currentInterface.getInetAddresses();
					while (inetAddresses.hasMoreElements()) {
						InetAddress inetAddress = inetAddresses.nextElement();
						if (NetUtil.isValidInetAdress(inetAddress)) {
							return currentInterface;
						}
					}
					networkInterface = currentInterface;
				} else if (networkInterface == null) {
					networkInterface = currentInterface;
				}
			}
		}
		return networkInterface;

	}

	/**
	 * Wait until an IP address is assigned.
	 *
	 * @param timeout
	 *            timeout to get the IP, 0 for unlimited.
	 *
	 * @throws IOException
	 *             if an IOException occurs.
	 * @see NetUtil#isValidInetAdress(byte[])
	 */
	private void waitUntilIpIsAssigned(int timeout) throws IOException {
		boolean validIp = false;

		long timeoutTimeMillis = Util.platformTimeMillis() + timeout;
		while (!validIp) {
			updateInterface();
			if (timeout != 0 && timeoutTimeMillis < Util.platformTimeMillis()) {
				throw new IOException(
						Messages.BUILDER.buildMessage(Level.SEVERE, Messages.WIFI_CATEGORY, Messages.ERROR_TIMEOUT));
			}

			Enumeration<InetAddress> inetAdresses = this.wifiInterface.getInetAddresses();
			while (inetAdresses.hasMoreElements()) {
				InetAddress inetAddress = inetAdresses.nextElement();
				if (NetUtil.isValidInetAdress(inetAddress)) {
					validIp = true;
					break;
				}
			}

			try {
				Thread.sleep(wifiIPAcquirePollingDelayMs);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}

	private IPConfiguration getDefaultIPConfiguration() {
		IPConfiguration clientIPConfiguration = new IPConfiguration();
		clientIPConfiguration.useDHCP(true);
		clientIPConfiguration.useStaticDNS(false);
		return clientIPConfiguration;
	}

	/**
	 * Initialises the network.
	 *
	 * @throws IOException
	 *             if something went wrong.
	 * @see NetworkInterfaceManager#start(NetworkInterface)
	 * @see NetworkInterfaceManager#enable(NetworkInterface)
	 */
	private void initNetwork() throws IOException {
		updateInterface();
		if (this.wifiInterface == null) {
			throw new IOException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.WIFI_CATEGORY,
					Messages.ERROR_NO_WIFI_MANAGER));
		}

		if (!NetworkInterfaceManager.isStarted(this.wifiInterface)) {
			NetworkInterfaceManager.start(this.wifiInterface);
		}

		if (!NetworkInterfaceManager.isEnabled(this.wifiInterface)) {
			NetworkInterfaceManager.enable(this.wifiInterface);
		}

		isSoftAPEnabled();
		this.configurationAP.setAccessPoint(this.wifiManager.getJoined());
	}
}
