/*
 * Java
 *
 * Copyright 2017-2025 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 */
package ej.junit;

public class CheckHelperTestListener implements TestListener {

	private static final String RESULT_PLACEHOLDER = "{}";

	private static final String DEFAULT_STATUS_PATTERN = ".:[|" + RESULT_PLACEHOLDER + "|]:.";

	private final boolean verbose;

	private final String passedTag;

	private final String failedTag;

	public CheckHelperTestListener() {
		this.verbose = Boolean.getBoolean("ej.junit.checkhelper.verbose");
		String statusPattern = System.getProperty("ej.junit.status.pattern", DEFAULT_STATUS_PATTERN);
		if (!statusPattern.contains(RESULT_PLACEHOLDER)) {
			statusPattern = DEFAULT_STATUS_PATTERN;
		}
		this.passedTag = statusPattern.replace(RESULT_PLACEHOLDER, "PASSED");
		this.failedTag = statusPattern.replace(RESULT_PLACEHOLDER, "FAILED");
	}

	@Override
	public void testBeforeClassMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.out.println("[JUNIT] Running @BeforeClass method: " + methodName);
		}
	}

	@Override
	public void testInstance(String testClass) {
		if (this.verbose) {
			System.out.println("[JUNIT] Instantiating: " + testClass);
		}
	}

	@Override
	public void testBeforeMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.out.println("[JUNIT] Running @Before method: " + methodName);
		}
	}

	@Override
	public void testAfterMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.out.println("[JUNIT] Running @After method: " + methodName);
		}
	}

	@Override
	public void testAfterClassMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.out.println("[JUNIT] Running @AfterClass method: " + methodName);
		}
	}

	private void log(String text) {
		System.out.println("[JUNIT] " + text);
	}

	@Override
	public void onSuiteStart(String name) {
		log("Suite start\t" + name);
	}

	@Override
	public void onSuiteEnd(String name, int runs, int failures, int errors) {
		log("Suite end\t" + name);
		System.err.println((failures == 0 && errors == 0 ? this.passedTag + ":\t" : this.failedTag + ":\t") + runs
				+ " tests\t" + failures + " failures\t" + errors + " errors");
	}

	@Override
	public void onCaseStart(String name, String className, String file, int line) {
		log("Test start\t" + name + "\t" + className + "\t" + file + "\t" + line);
	}

	@Override
	public void onCaseEnd(String name, double duration) {
		log("Test end\t" + name + "\t" + duration + " seconds");

	}

	@Override
	public void onFailure(String message, String type, String trace) {
		log("Failure\t" + type + "\t" + message);
		log("TRACE START\n" + trace);
		log("TRACE END");
	}

	@Override
	public void onError(String message, String type, String trace) {
		log("Error\t" + type + "\t" + message);
		log("TRACE START\n" + trace);
		log("TRACE END");
	}

	@Override
	public void onSkip(String message) {
		log("Skipped\t" + message);
	}
}
