/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http.encoding;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import ej.hoka.http.HttpRequest;
import ej.hoka.http.HttpResponse;
import ej.hoka.http.HttpServer;

/**
 * Interface for defining HTTP transfer coding handlers.
 * <p>
 * The HTTP transfer coding handler decodes data from the body of a {@link HttpRequest} and encodes the body of a
 * {@link HttpResponse}.
 * <p>
 * Transfer coding is specified in <code>transfer-encoding</code> HTTP header.
 * <p>
 * Encoding handlers should be registered in the {@link HttpServer} in order to use them.
 *
 * @see EncodingRegistry#registerTransferEncoding(TransferEncoding)
 *
 */
public interface TransferEncoding {

	/**
	 * Returns the supported encoding id.
	 *
	 * @return an internal {@link String} in lower case format.
	 */
	String getId();

	/**
	 * Opens an {@link InputStream} that can be used to decode message body of the given request. The returned
	 * {@link InputStream} MUST conforms to the followings:
	 * <ul>
	 * <li>The {@link InputStream} MUST reach the EOF (i.e. read methods returns <code>-1</code>) when the request body
	 * has been completely read.</li>
	 * <li>The {@link InputStream#close()} method MUST read any remaining bytes from the message body (if any) and MUST
	 * NOT close the underlying stream.</li>
	 * </ul>
	 *
	 * @param request
	 *            the {@link HttpRequest} to be decoded by this transfer coding handler.
	 * @param input
	 *            the {@link InputStream} from which encoded message body can be read.
	 * @return the {@link InputStream} used to decode message body of the given request.
	 * @throws IOException
	 *             if any I/O Error occurs.
	 */
	InputStream open(HttpRequest request, InputStream input) throws IOException;

	/**
	 * Opens an {@link OutputStream} that can be used to encode the message body of the {@link HttpResponse}.
	 *
	 * @param response
	 *            the {@link HttpResponse} to be encoded by this transfer coding handler.
	 * @param output
	 *            the {@link OutputStream} where the encoded message body is written.
	 * @return the output stream used to encode message body of the given response.
	 * @throws IOException
	 *             if any I/O Error occurs.
	 */
	OutputStream open(HttpResponse response, OutputStream output) throws IOException;

}
