/*
 * Java
 *
 * Copyright 2009-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp.widget;

import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.fp.util.ThirdEventThread;
import ej.fp.util.ThirdEventThread.ThirdEventWidget;
import ej.microui.event.EventButton;

/**
 * Extended button widget which manages <i>long</i> button press events.
 * <p>
 * Notes:
 * <ul>
 * <li>This widget requires a listener: an implementation of {@link LongButtonListener} to send the button events,</li>
 * <li>The widget label must be an integer. It allows to create a MicroUI event using this button ID.</li>
 * </ul>
 */
@SuppressWarnings("nls")
@WidgetDescription(attributes = { @WidgetAttribute(name = "label"), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "skin"), @WidgetAttribute(name = "pushedSkin"),
		@WidgetAttribute(name = "longPeriod", isOptional = true), @WidgetAttribute(name = "filter", isOptional = true),
		@WidgetAttribute(name = "listenerClass", isOptional = true) })
public class LongButton extends Button implements ThirdEventWidget {

	/**
	 * Interface that handle long button press events.
	 */
	public static interface LongButtonListener extends ButtonListener {

		/**
		 * The specified button has been pressed for a long time.
		 *
		 * @param widget
		 *            the button that has been pressed for a long time.
		 */
		void pressLong(LongButton widget);
	}

	/**
	 * Default implementation of {@link LongButtonListener}.
	 *
	 * @see ej.fp.widget.Button.ButtonListenerToButtonEvents
	 */
	public static class LongButtonListenerToButtonEvents extends ButtonListenerToButtonEvents
			implements LongButtonListener {

		@Override
		public void pressLong(LongButton widget) {
			EventButton.sendLongEvent(getMicroUIGeneratorTag(), widget.getID());
		}
	}

	private final ThirdEventThread longThread;
	private int longPeriod;

	/**
	 * Creates a long button widget.
	 * <p>
	 * The default long period is 500ms.
	 */
	public LongButton() {
		this.longThread = new ThirdEventThread(this, false);
		this.longPeriod = 500;
	}

	/**
	 * Set the long event period.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param longPeriod
	 *            a time in milliseconds.
	 */
	public void setLongPeriod(int longPeriod) {
		if (longPeriod <= 0) {
			throw new IllegalArgumentException("Long period cannot be negative.");
		}
		this.longPeriod = longPeriod;
	}

	@Override
	public void setListenerClass(String listenerClassName) {
		setListenerClass(LongButtonListener.class, listenerClassName);
	}

	@Override
	public void start() {
		super.start();
		this.longThread.start();
	}

	@Override
	public void dispose() {
		this.longThread.dispose();
		super.dispose();
	}

	@Override
	protected Object newDefaultListener() {
		return new LongButtonListenerToButtonEvents();
	}

	@Override
	public void sendThirdEvent() {
		((LongButtonListener) getListener()).pressLong(this);
	}

	@Override
	public int getSleepPeriod() {
		return this.longPeriod;
	}

	@Override
	protected void press() {
		ThirdEventThread longThread = this.longThread;
		if (!longThread.isSendingThirdEvent()) {
			super.press();
			longThread.wakeup();
		}
		// else don't send again a press event if we are sending repeat events
	}

	@Override
	protected void release() {
		this.longThread.goToSleep();
		super.release();
	}
}