/*
 * Java
 *
 * Copyright 2009-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp.widget;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.MouseListener;
import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.fp.util.ThirdEventThread;
import ej.fp.util.ThirdEventThread.ThirdEventWidget;
import ej.fp.util.WidgetWithListener;
import ej.fp.widget.Button.ButtonListener;
import ej.microui.event.EventCommand;

/**
 * This widget simulates until five buttons organized like a cross + center button. It manages three states by button:
 * pressed, repeat and released.
 * <p>
 * Notes:
 * <ul>
 * <li>This widget requires a listener: an implementation of {@link ButtonListener} to send the button events,</li>
 * <li>The widget label must be an integer. It allows to create a MicroUI event using this button ID,</li>
 * <li>The center button event is sent using the mouse third button (often right button),</li>
 * <li>The center button can be disabled using {@link #setDisableEnter(boolean)} method.</li>
 * </ul>
 */
@SuppressWarnings("nls")
@WidgetDescription(attributes = { @WidgetAttribute(name = "label", isOptional = true), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "skin"),
		@WidgetAttribute(name = "filter", isOptional = true), @WidgetAttribute(name = "upSkin"),
		@WidgetAttribute(name = "downSkin"), @WidgetAttribute(name = "leftSkin"), @WidgetAttribute(name = "rightSkin"),
		@WidgetAttribute(name = "enterSkin", isOptional = true),
		@WidgetAttribute(name = "disableEnter", isOptional = true),
		@WidgetAttribute(name = "repeatPeriod", isOptional = true),
		@WidgetAttribute(name = "listenerClass", isOptional = true) })
public class Joystick extends WidgetWithListener implements MouseListener, ThirdEventWidget {

	/**
	 * Interface that handle joystick buttons events.
	 */
	public static interface JoystickListener {

		/**
		 * The up button has been pressed.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed.
		 */
		void pressUp(Joystick widget);

		/**
		 * The down button has been pressed.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed.
		 */
		void pressDown(Joystick widget);

		/**
		 * The left button has been pressed.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed.
		 */
		void pressLeft(Joystick widget);

		/**
		 * The right button has been pressed.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed.
		 */
		public void pressRight(Joystick widget);

		/**
		 * The enter button has been pressed.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed.
		 */
		void pressEnter(Joystick widget);

		/**
		 * The up button has been held down.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed again.
		 */
		void repeatUp(Joystick widget);

		/**
		 * The down button has been held down.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed again.
		 */
		void repeatDown(Joystick widget);

		/**
		 * The left button has been held down.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed again.
		 */
		void repeatLeft(Joystick widget);

		/**
		 * The right button has been held down.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed again.
		 */
		void repeatRight(Joystick widget);

		/**
		 * The enter button has been held down.
		 *
		 * @param widget
		 *            the joystick where the button has been pressed again.
		 */
		void repeatEnter(Joystick widget);

		/**
		 * The up button has been released.
		 *
		 * @param widget
		 *            the joystick where the button has been released.
		 */
		void releaseUp(Joystick widget);

		/**
		 * The down button has been released.
		 *
		 * @param widget
		 *            the joystick where the button has been released.
		 */
		void releaseDown(Joystick widget);

		/**
		 * The left button has been released.
		 *
		 * @param widget
		 *            the joystick where the button has been released.
		 */
		void releaseLeft(Joystick widget);

		/**
		 * The right button has been released.
		 *
		 * @param widget
		 *            the joystick where the button has been released.
		 */
		void releaseRight(Joystick widget);

		/**
		 * The enter button has been released.
		 *
		 * @param widget
		 *            the joystick where the button has been released.
		 */
		void releaseEnter(Joystick widget);
	}

	/**
	 * Default implementation of {@link JoystickListener}.
	 * <p>
	 * This implementation sends some MicroUI Command events. It targets the common MicroUI Command generator identified
	 * by the tag {@link EventCommand#COMMON_MICROUI_GENERATOR_TAG}.
	 * <p>
	 * No event is sent on release states (only press and repeat).
	 * <p>
	 * This tag can be changed overriding the method {@link #getMicroUIGeneratorTag()}.
	 */
	public static class JoystickListenerToCommandEvents implements JoystickListener {

		@Override
		public void pressDown(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.DOWN);
		}

		@Override
		public void pressEnter(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.SELECT);
		}

		@Override
		public void pressLeft(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.LEFT);
		}

		@Override
		public void pressRight(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.RIGHT);
		}

		@Override
		public void pressUp(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.UP);
		}

		@Override
		public void releaseDown(Joystick widget) {
			// no release event
		}

		@Override
		public void releaseEnter(Joystick widget) {
			// no release event
		}

		@Override
		public void releaseLeft(Joystick widget) {
			// no release event
		}

		@Override
		public void releaseRight(Joystick widget) {
			// no release event
		}

		@Override
		public void releaseUp(Joystick widget) {
			// no release event
		}

		@Override
		public void repeatDown(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.DOWN);
		}

		@Override
		public void repeatEnter(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.SELECT);
		}

		@Override
		public void repeatLeft(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.LEFT);
		}

		@Override
		public void repeatRight(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.RIGHT);
		}

		@Override
		public void repeatUp(Joystick widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.UP);
		}

		/**
		 * Gets the MicroUI Command events generator tag. This generator has to match the generator set during the
		 * MicroEJ platform build in <code>microui/microui.xml</code>
		 *
		 * @return a MicroUI Command events generator tag
		 */
		protected String getMicroUIGeneratorTag() {
			return EventCommand.COMMON_MICROUI_GENERATOR_TAG;
		}
	}

	private final ThirdEventThread repeatThread;
	private final JoystickButton up;
	private final JoystickButton down;
	private final JoystickButton left;
	private final JoystickButton right;
	private final JoystickButton enter;

	private JoystickListener listener;
	private int centerX;
	private int centerY;
	private int period;
	private boolean disableEnter;

	/**
	 * Creates a joystick widget.
	 * <p>
	 * The default repeat period is 200ms.
	 */
	public Joystick() {
		this.up = new Up();
		this.down = new Down();
		this.left = new Left();
		this.right = new Right();
		this.enter = new Enter();
		this.repeatThread = new ThirdEventThread(this, true);
		this.period = 200;
	}

	/**
	 * Sets the skin to show joystick UP action.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param skin
	 *            skin for joystick UP action.
	 */
	public void setUpSkin(Image skin) {
		this.up.skin = skin;
	}

	/**
	 * Sets the skin to show joystick DOWN action.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param skin
	 *            skin for joystick DOWN action.
	 */
	public void setDownSkin(Image skin) {
		this.down.skin = skin;
	}

	/**
	 * Sets the skin to show joystick LEFT action.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param skin
	 *            skin for joystick LEFT action.
	 */
	public void setLeftSkin(Image skin) {
		this.left.skin = skin;
	}

	/**
	 * Sets the skin to show joystick RIGHT action.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param skin
	 *            skin for joystick RIGHT action.
	 */
	public void setRightSkin(Image skin) {
		this.right.skin = skin;
	}

	/**
	 * Sets the skin to show joystick ENTER action.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param skin
	 *            skin for joystick ENTER action.
	 */
	public void setEnterSkin(Image skin) {
		this.enter.skin = skin;
	}

	/**
	 * Sets the repeat event period.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param period
	 *            a time in milliseconds.
	 */
	public void setRepeatPeriod(int period) {
		if (period <= 20) {
			throw new IllegalArgumentException("Joystick period cannot be smaller than 20ms.");
		}
		this.period = period;
	}

	/**
	 * Disables <i>enter</i> button. By default this button is enabled.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param disableEnter
	 *            true to disable enter button.
	 */
	public void setDisableEnter(boolean disableEnter) {
		this.disableEnter = disableEnter;
	}

	/**
	 * Defines the user class which has to implement {@link JoystickListener}.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param listenerClassName
	 *            user listener class name.
	 */
	public void setListenerClass(String listenerClassName) {
		setListenerClass(JoystickListener.class, listenerClassName);
	}

	@Override
	public synchronized void finalizeConfiguration() {
		super.finalizeConfiguration();
		this.centerX = getWidth() / 2;
		this.centerY = getHeight() / 2;
	}

	@Override
	public void start() {
		super.start();
		this.listener = newListener(JoystickListener.class);
		this.repeatThread.start();
	}

	@Override
	public void dispose() {
		this.repeatThread.dispose();
		this.up.dispose();
		this.down.dispose();
		this.left.dispose();
		this.right.dispose();
		this.enter.dispose();
		super.dispose();
	}

	@Override
	protected Object newDefaultListener() {
		return new JoystickListenerToCommandEvents();
	}

	@Override
	public void sendThirdEvent() {
		if (!this.enter.sendRepeat(this)) {
			this.up.sendRepeat(this);
			this.down.sendRepeat(this);
			this.left.sendRepeat(this);
			this.right.sendRepeat(this);
		}
	}

	/**
	 * The thread sleep period is different than buttons period: thread should work faster than the buttons period in
	 * order to be able to simulate several repeat event on several buttons.
	 */
	@Override
	public int getSleepPeriod() {
		return 20;
	}

	@Override
	public void mousePressed(int x, int y, MouseButton button) {
		// check the mouse button is the first

		if (button == MouseButton.FIRST_BUTTON) {

			int relativeX = this.centerX - x;
			int relativeY = this.centerY - y;

			// get mouse direction
			boolean mouseEquals = relativeX == relativeY || relativeX == -relativeY;
			boolean mouseRight = relativeX < 0 && relativeY < -relativeX && relativeY > relativeX;
			boolean mouseLeft = relativeX > 0 && relativeY < relativeX && relativeY > -relativeX;
			boolean mouseDown = relativeY < 0 && relativeX < -relativeY && relativeX > relativeY;
			boolean mouseUp = relativeY > 0 && relativeX < relativeY && relativeX > -relativeY;

			// set the right skin
			if (!mouseEquals) {
				if (mouseRight) {
					this.right.sendPress(this);
				} else if (mouseLeft) {
					this.left.sendPress(this);
				} else if (mouseUp) {
					this.up.sendPress(this);
				} else if (mouseDown) {
					this.down.sendPress(this);
				}
				// launch repeat thread
				launchThread();
			}
		} else if (button == MouseButton.THIRD_BUTTON && !this.disableEnter) {
			this.enter.sendPress(this);
			// launch repeat thread
			launchThread();
		}
	}

	@Override
	public void mouseReleased(int x, int y, MouseButton button) {

		if (button == MouseButton.FIRST_BUTTON) {
			this.up.sendRelease(this);
			this.down.sendRelease(this);
			this.left.sendRelease(this);
			this.right.sendRelease(this);
		} else if (button == MouseButton.THIRD_BUTTON && !this.disableEnter) {
			this.enter.sendRelease(this);
		}

		if (!this.up.sendThirdEvent && !this.down.sendThirdEvent && !this.left.sendThirdEvent
				&& !this.right.sendThirdEvent && !this.enter.sendThirdEvent) {
			stopThread();
		}
	}

	private void launchThread() {
		this.repeatThread.wakeup();
	}

	private void stopThread() {
		this.repeatThread.goToSleep();
	}

	private abstract static class JoystickButton {
		private Image skin;
		private boolean sendThirdEvent; // false init
		private long time; // 0 init

		abstract void press(Joystick joystick);

		abstract void repeat(Joystick joystick);

		abstract void release(Joystick joystick);

		private void sendPress(Joystick joystick) {
			this.sendThirdEvent = true;
			this.time = System.currentTimeMillis();
			if (this.skin != null) {
				joystick.setCurrentSkin(this.skin);
			}
			press(joystick);
		}

		private boolean sendRepeat(Joystick joystick) {
			if ((this.sendThirdEvent && (System.currentTimeMillis() - this.time >= joystick.period))) {
				this.time = System.currentTimeMillis();
				repeat(joystick);
				return true;
			}
			return false;
		}

		private void sendRelease(Joystick joystick) {
			if (this.sendThirdEvent) {
				this.sendThirdEvent = false;
				release(joystick);
				joystick.setCurrentSkin(joystick.getSkin());
			}
		}

		private void dispose() {
			FrontPanel.getFrontPanel().disposeIfNotNull(this.skin);
		}
	}

	private static class Up extends JoystickButton {
		@Override
		void press(Joystick joystick) {
			joystick.listener.pressUp(joystick);
		}

		@Override
		void repeat(Joystick joystick) {
			joystick.listener.repeatUp(joystick);
		}

		@Override
		void release(Joystick joystick) {
			joystick.listener.releaseUp(joystick);
		}
	}

	private static class Down extends JoystickButton {
		@Override
		void press(Joystick joystick) {
			joystick.listener.pressDown(joystick);
		}

		@Override
		void repeat(Joystick joystick) {
			joystick.listener.repeatDown(joystick);
		}

		@Override
		void release(Joystick joystick) {
			joystick.listener.releaseDown(joystick);
		}
	}

	private static class Left extends JoystickButton {
		@Override
		void press(Joystick joystick) {
			joystick.listener.pressLeft(joystick);
		}

		@Override
		void repeat(Joystick joystick) {
			joystick.listener.repeatLeft(joystick);
		}

		@Override
		void release(Joystick joystick) {
			joystick.listener.releaseLeft(joystick);
		}
	}

	private static class Right extends JoystickButton {
		@Override
		void press(Joystick joystick) {
			joystick.listener.pressRight(joystick);
		}

		@Override
		void repeat(Joystick joystick) {
			joystick.listener.repeatRight(joystick);
		}

		@Override
		void release(Joystick joystick) {
			joystick.listener.releaseRight(joystick);
		}
	}

	private static class Enter extends JoystickButton {
		@Override
		void press(Joystick joystick) {
			joystick.listener.pressEnter(joystick);
		}

		@Override
		void repeat(Joystick joystick) {
			joystick.listener.repeatEnter(joystick);
		}

		@Override
		void release(Joystick joystick) {
			joystick.listener.releaseEnter(joystick);
		}
	}
}
