/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util;

/**
 * A builder of a bluetooth payload.
 */
public class BluetoothPayloadBuilder {

	private byte[] payload;

	/**
	 * Instatiates a payload without any data.
	 */
	public BluetoothPayloadBuilder() {
		this.payload = new byte[0];
	}

	/**
	 * Instantiates a payload builder with a first value.
	 *
	 * @param type
	 *            the type of payload to add.
	 * @param value
	 *            the value of the payload to add.
	 */
	public BluetoothPayloadBuilder(byte type, byte[] value) {
		this();
		append(type, value);
	}

	/**
	 * Appends a byte value. If the type already exist overrides it.
	 *
	 * @param type
	 *            the type of payload to add.
	 * @param value
	 *            the value of the payload to add.
	 */
	public BluetoothPayloadBuilder(byte type, byte value) {
		this();
		append(type, value);
	}

	/**
	 * Appends a String value. If the type already exist overrides it.
	 *
	 * @param type
	 *            the type of payload to add.
	 * @param value
	 *            the value of the payload to add.
	 */
	public BluetoothPayloadBuilder(byte type, String value) {
		this();
		append(type, value);
	}

	/**
	 * Appends a byte array payload. If the type already exist overrides it.
	 *
	 * @param type
	 *            the type of payload to add.
	 * @param value
	 *            the value of the payload to add.
	 * @return the payload builder.
	 */
	public synchronized BluetoothPayloadBuilder append(byte type, byte[] value) {
		this.payload = BluetoothPayloadUtil.appendPayload(type, value,
				BluetoothPayloadUtil.removePayload(type, this.payload));
		return this;
	}

	/**
	 * Appends a byte payload. If the type already exist overrides it.
	 *
	 * @param type
	 *            the type of payload to add.
	 * @param value
	 *            the value of the payload to add.
	 * @return the payload builder.
	 */
	public BluetoothPayloadBuilder append(byte type, byte value) {
		byte[] bytes = new byte[1];
		bytes[0] = value;
		return append(type, bytes);
	}

	/**
	 * Appends a String payload. If the type already exist, overrides it.
	 *
	 * @param type
	 *            the type of payload to add.
	 * @param value
	 *            the value of the payload to add.
	 * @return the payload builder.
	 */
	public BluetoothPayloadBuilder append(byte type, String value) {
		return append(type, value.getBytes());
	}

	/**
	 * Appends a String payload. If the type already exist, overrides it.
	 *
	 * @param type
	 *            the type of payload to remove.
	 * @return the payload builder.
	 */
	public synchronized BluetoothPayloadBuilder remove(byte type) {
		this.payload = BluetoothPayloadUtil.removePayload(type, this.payload);
		return this;
	}

	/**
	 * Gets the payload, this is not a copy of the payload.
	 *
	 * @return the payload.
	 */
	public byte[] getPayload() {
		return this.payload;
	}

}
