/*
 * Java
 *
 * Copyright 2015-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.event.Event;
import ej.microui.event.generator.Buttons;
import ej.microui.event.generator.Pointer;
import ej.microui.util.EventHandler;

/**
 * Dispatches the event received on a desktop to its children.
 * <ul>
 * <li>Pointer events are grouped in sessions. A session starts when the pointer is pressed, and ends when the pointer
 * is released.
 * <p>
 * While no renderable consumes the events, they are sent to the renderable that is under the pointer (see
 * {@link Desktop#panelAt(int, int)} and {@link Panel#getWidgetAt(int, int)}), then sent to all its parent hierarchy.
 * <p>
 * Once a renderable has consumed an event, it will be the only one to receive the next events during the session.</li>
 * <li>For other events, the events are sent to the focus owner (see {@link Widget#hasFocus()}), then sent to all its
 * parent hierarchy while the event is not consumed.</li>
 * </ul>
 *
 * @since 2.0
 */
public class DesktopEventManagementPolicy implements EventHandler {

	@NonNull
	private final Desktop desktop;

	/**
	 * Creates a desktop event manager.
	 *
	 * @param desktop
	 *            the desktop to dispatch in.
	 */
	public DesktopEventManagementPolicy(@NonNull Desktop desktop) {
		this.desktop = desktop;
	}

	@Override
	public boolean handleEvent(int event) {
		// dispatch events (separate pointer and other events)
		int type = Event.getType(event);
		if (type == Event.POINTER) {
			pointerEvent(event);
		} else {
			Panel panel = this.desktop.getActivePanel();
			if (panel != null) {
				if (this.desktop.isAccessible(panel)) {
					Widget focus = panel.getFocus();
					sendEventHierarchy(panel, focus, event);
				}
			} else { // directly send to desktop
				this.desktop.handleEvent(event);
			}
		}
		return false;
	}

	/**
	 * Handles a pointer button event.<br>
	 * This method is synchronized by microUI display pump.
	 */
	private void pointerEvent(int event) {
		@NonNull
		Pointer pointer = (Pointer) Event.getGenerator(event);
		ExtendedPointer xPointer = ExtendedPointer.getXPointer(pointer);
		int action = Buttons.getAction(event);

		Renderable renderablePressed = xPointer.getRenderablePressed();
		updatePointer(xPointer, pointer);
		Renderable renderableUnder = xPointer.getRenderableUnder();
		switch (action) {
		case Pointer.DRAGGED:
			dragged(event, pointer, xPointer, renderablePressed, renderableUnder);
			break;
		case Pointer.MOVED:
			// move renderable under
			moved(event, renderableUnder);
			break;
		case Buttons.PRESSED:
			pressed(event, pointer, xPointer, renderableUnder);
			break;
		case Buttons.RELEASED:
			released(event, pointer, xPointer, renderablePressed);
			break;
		}
	}

	private void dragged(int event, @NonNull Pointer pointer, @NonNull ExtendedPointer xPointer,
			@Nullable Renderable renderablePressed, @Nullable Renderable renderableUnder) {
		if (renderablePressed != null) {
			// dragged pressed renderable
			sendEventToRenderable(renderablePressed, event);
		} else {
			Renderable renderable = sendEventToRenderableHierarchy(renderableUnder, event);
			if (renderable != null) {
				xPointer.setRenderablePressed(renderable);

				sendExit(pointer, renderableUnder, renderable);
			}
		}
	}

	private void moved(int event, @Nullable Renderable renderableUnder) {
		sendEventToRenderableHierarchy(renderableUnder, event);
	}

	private void pressed(int event, @NonNull Pointer pointer, @NonNull ExtendedPointer xPointer,
			@Nullable Renderable renderableUnder) {
		if (renderableUnder != null) {
			Panel panelUnder = getPanel(renderableUnder);
			this.desktop.setActivePanelInternal(panelUnder);

			Renderable renderablePressed = sendEventToRenderableHierarchy(renderableUnder, event);
			if (renderablePressed != null) {
				xPointer.setRenderablePressed(renderablePressed);
				focusRenderable(renderablePressed);

				sendExit(pointer, renderableUnder, renderablePressed);
			}
		}
	}

	@Nullable
	private Panel getPanel(@NonNull Renderable renderable) {
		if (renderable instanceof Widget) {
			Widget widget = (Widget) renderable;
			return widget.getPanel();
		} else if (renderable instanceof Panel) {
			return (Panel) renderable;
		}
		return null;
	}

	private void released(int event, @NonNull Pointer pointer, @NonNull ExtendedPointer xPointer,
			@Nullable Renderable renderablePressed) {
		if (pointer.buttonsState == 0) {
			xPointer.setRenderablePressed(null);
		}
		if (renderablePressed != null) {
			sendEventToRenderable(renderablePressed, event);
		} else {
			Renderable renderableUnder = xPointer.getRenderableUnder();
			renderablePressed = sendEventToRenderableHierarchy(renderableUnder, event);
			if (renderablePressed != null) {
				focusRenderable(renderablePressed);

				sendExit(pointer, renderableUnder, renderablePressed);
			}
		}
	}

	private void focusRenderable(@Nullable Renderable renderablePressed) {
		if (renderablePressed != null) {
			if (renderablePressed instanceof Widget) {
				Widget widget = (Widget) renderablePressed;
				Panel panel = widget.getPanel();
				if (panel != null) {
					panel.setFocus(widget);
				}
			}
		}
	}

	/**
	 * Updates a pointer.<br>
	 * Search the widget that is under the pointer coordinates.<br>
	 * Generates {@link Pointer#ENTERED} or {@link Pointer#EXITED} events if necessary.
	 */
	private void updatePointer(@NonNull ExtendedPointer xPointer, @NonNull Pointer pointer) {
		int x = pointer.getX();
		int y = pointer.getY();

		Renderable oldRenderable = xPointer.getRenderableUnder();
		// search widget under the mouse
		Panel panel = this.desktop.panelAt(x, y);
		Renderable renderable;
		if (panel != null && this.desktop.isAccessible(panel)) {
			Widget widget = panel.getWidgetAt(x, y);
			// search the first enabled widget in the hierarchy
			while (widget != null && !widget.isEnabled()) {
				widget = widget.getParent();
			}
			if (widget != null) {
				renderable = widget;
			} else {
				// no enabled widget at this position in the panel
				renderable = panel;
			}
		} else {
			// no accessible panel at this position in the desktop
			renderable = this.desktop;
		}

		// update the widget under mouse
		xPointer.setRenderableUnder(renderable);

		if (oldRenderable != renderable) {
			// renderable under has changed
			// exit old
			// check still visible
			if (oldRenderable != null && oldRenderable.isShown()) {
				sendExitToRenderableHierarchy(oldRenderable, pointer, x, y);
			}
			// enter new
			sendEnterToRenderableHierarchy(renderable, oldRenderable, pointer, x, y);
		}

	}

	/**
	 * Sends a pointer exit to the widgets under the pointer that didn't capture the pointer session.
	 *
	 * @param pointer
	 *            the pointer.
	 * @param renderableUnder
	 *            the renderable under (leaf) the pointer.
	 * @param renderablePressed
	 *            the renderable that just captured the pointer session.
	 */
	private void sendExit(@NonNull Pointer pointer, @Nullable Renderable renderableUnder,
			@Nullable Renderable renderablePressed) {
		// Renderable under will not receive other events -> consider it exited.
		int event = buildEvent(pointer, Pointer.EXITED);
		Widget widget;
		Panel panel;
		if (renderableUnder instanceof Widget) {
			widget = (Widget) renderableUnder;
			panel = widget.getPanel();
		} else if (renderableUnder instanceof Panel) {
			widget = null;
			panel = (Panel) renderableUnder;
		} else {
			return;
		}
		while (widget != null) {
			if (widget == renderablePressed) {
				return;
			}
			// Do not check if the widget is still under the pointer.
			if (widget.isEnabled()) {
				widget.handleEvent(event);
			}
			// Continue recursively up to the renderable pressed.
			widget = widget.getParent();
		}
		if (panel != renderablePressed && panel != null) {
			panel.handleEvent(event);
		}
	}

	/**
	 * Builds and sends a pointer exit event while exiting old renderables.
	 */
	private void sendExitToRenderableHierarchy(@NonNull Renderable renderable, @NonNull Pointer pointer, int x, int y) {
		if (renderable instanceof Widget) {
			Widget widget = (Widget) renderable;
			sendExitEventHierarchy(widget.getPanel(), widget, pointer, x, y);
		} else if (renderable instanceof Panel) {
			sendExitEventHierarchy((Panel) renderable, null, pointer, x, y);
		}
	}

	/**
	 * Builds and sends a pointer exit event while exiting old renderables.
	 */
	private void sendExitEventHierarchy(@Nullable Panel panel, @Nullable Widget widget, @NonNull Pointer pointer, int x,
			int y) {
		int event = buildEvent(pointer, Pointer.EXITED);
		while (widget != null) {
			// Check if the widget is still under the pointer.
			if (widget.isShown() && widget.contains(x - widget.getAbsoluteXInternal() + widget.x,
					y - widget.getAbsoluteYInternal() + widget.y)) {
				return;
			} else {
				if (widget.isEnabled()) {
					widget.handleEvent(event);
				}
			}
			// search recursively
			widget = widget.getParent();
		}
		// Send to panel if not anymore under the pointer.
		if (panel != null && !panel.contains(x, y)) {
			panel.handleEvent(event);
		}
	}

	/**
	 * Builds and sends a pointer enter event while entering new renderables.
	 */
	private void sendEnterToRenderableHierarchy(@Nullable Renderable renderable, @Nullable Renderable oldRenderable,
			@NonNull Pointer pointer, int x, int y) {
		Panel panel;
		Widget widget;
		if (renderable instanceof Widget) {
			widget = (Widget) renderable;
			panel = widget.panel;
		} else if (renderable instanceof Panel) {
			panel = (Panel) renderable;
			widget = null;
		} else {
			return;
		}
		Panel oldPanel;
		Widget oldWidget;
		if (oldRenderable instanceof Widget) {
			oldWidget = (Widget) oldRenderable;
			oldPanel = oldWidget.getPanel();
		} else if (oldRenderable instanceof Panel) {
			oldPanel = (Panel) oldRenderable;
			oldWidget = null;
		} else {
			oldPanel = null;
			oldWidget = null;
		}
		sendEnterEventHierarchy(panel, widget, oldPanel, oldWidget, pointer, x, y);
	}

	/**
	 * Builds and sends a pointer enter event while entering new renderables.
	 */
	private void sendEnterEventHierarchy(@Nullable Panel panel, @Nullable Widget widget, @Nullable Panel oldPanel,
			@Nullable Widget oldWidget, @NonNull Pointer pointer, int x, int y) {
		int event = buildEvent(pointer, Pointer.ENTERED);
		while (widget != null) {
			// Check already in "old" hierarchy.
			Widget widgetCandidate = oldWidget;
			while (widgetCandidate != null) {
				if (widget == widgetCandidate) {
					return;
				}
				widgetCandidate = widgetCandidate.getParent();
			}
			if (widget.isEnabled()) {
				widget.handleEvent(event);
			}
			// search recursively
			widget = widget.getParent();
		}
		// send to panel
		if (panel != null && panel != oldPanel) {
			panel.handleEvent(event);
		}
	}

	private int buildEvent(@NonNull Pointer pointer, int action) {
		int event = Event.buildEvent(pointer.getEventType(), pointer, Pointer.buildEventData(action, 0));
		return event;
	}

	/**
	 * Sends the event to the widget and recursively to its parents, then to its panel, then to the desktop (this) while
	 * not consumed.
	 */
	private Renderable sendEventHierarchy(@Nullable Panel panel, @Nullable Widget widget, int event) {
		while (widget != null) {
			if (widget.isEnabled() && widget.handleEvent(event)) {
				return widget;
			}
			// search recursively
			widget = widget.getParent();
		}
		// send to panel
		if (panel != null && panel.handleEvent(event)) {
			return panel;
		}
		// send to desktop
		if (this.desktop.handleEvent(event)) {
			return this.desktop;
		}
		return null;
	}

	/**
	 * Sends the event to the renderable and recursively to its parents (widget -> composite* -> panel -> desktop) while
	 * not consumed.
	 */
	private Renderable sendEventToRenderableHierarchy(@Nullable Renderable renderable, int event) {
		if (renderable instanceof Widget) {
			Widget widget = (Widget) renderable;
			return sendEventHierarchy(widget.getPanel(), widget, event);
		} else if (renderable instanceof Panel) {
			return sendEventHierarchy((Panel) renderable, null, event);
		} else {
			return sendEventHierarchy(null, null, event);
		}
	}

	/**
	 * Sends a event to a renderable verifying that the renderable is not <code>null</code>.
	 */
	private void sendEventToRenderable(@Nullable Renderable renderable, int event) {
		if (renderable != null) {
			renderable.handleEvent(event);
		}
	}
}
