/*
 * Java
 *
 * Copyright 2014-2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.rcommand.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import ej.rcommand.CommandSender;
import ej.rcommand.RemoteConnection;

/**
 * Implements an {@link RemoteConnection} that wraps another {@link RemoteConnection} that can change dynamically.
 */
public class DynamicRemoteConnection implements RemoteConnection {

	/**
	 * Set during {@link #setConnection(RemoteConnection)}.
	 */
	private volatile RemoteConnection connection;

	private boolean closed = false;

	/**
	 * Set the underlying connection. If defined, previous connection is closed.<br>
	 * Given newConnection can be null.
	 */
	public void setConnection(RemoteConnection newConnection) {
		synchronized (this) {
			RemoteConnection previousConnection = this.connection;
			this.connection = newConnection;
			if (previousConnection != null) {
				previousConnection.close();
			}
			this.notifyAll();
		}
	}

	@Override
	public String readCommand() throws IOException {
		return waitForConnection().readCommand();
	}

	private RemoteConnection waitForConnection() throws IOException {
		synchronized (this) {
			while (this.connection == null) {
				if (this.closed) {
					throw new IOException("Connection closed");
				}
				try {
					this.wait();
				} catch (InterruptedException e) {
				}
			}
			return this.connection;
		}
	}

	@Override
	public String readString() throws IOException {
		return waitForConnection().readString();
	}

	@Override
	public long readLong() throws IOException {
		return waitForConnection().readLong();
	}

	@Override
	public byte[] readByteArray() throws IOException {
		return waitForConnection().readByteArray();
	}

	@Override
	public InputStream readByteArrayAsInputStream() throws IOException {
		return waitForConnection().readByteArrayAsInputStream();
	}

	@Override
	public void startCommand(String command) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.startCommand(command);
		}
	}

	@Override
	public void sendString(String s) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendString(s);
		}
	}

	@Override
	public void sendLong(long l) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendLong(l);
		}
	}

	@Override
	public void sendByteArray(byte[] a) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendByteArray(a);
		}
	}

	@Override
	public void sendByteArrayAsInputStream(InputStream is) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendByteArrayAsInputStream(is);
		}
	}

	@Override
	public void flushCommand() {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.flushCommand();
		}
	}

	@Override
	public void close() {
		synchronized (this) {
			this.closed = true;
			RemoteConnection connection = this.connection;
			if (connection != null) {
				connection.close();
			}
			this.notifyAll();
		}
	}

	@Override
	public void skipParameters() throws IOException {
		waitForConnection().skipParameters();
	}

	@Override
	public List<Object> readParameters() throws IOException {
		return waitForConnection().readParameters();
	}

	@Override
	public void sendParams(List<Object> params) {
		RemoteConnection connection = this.connection;
		if (connection != null) {
			connection.sendParams(params);
		}
	}

	@Override
	public void sendInt(int i) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendInt(i);
		}
	}

	@Override
	public void sendFloat(float f) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendFloat(f);
		}
	}

	@Override
	public void sendDouble(double d) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendDouble(d);
		}
	}

	@Override
	public void sendBoolean(boolean b) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendBoolean(b);
		}
	}

	@Override
	public int readInt() throws IOException {
		return waitForConnection().readInt();
	}

	@Override
	public float readFloat() throws IOException {
		return waitForConnection().readFloat();
	}

	@Override
	public double readDouble() throws IOException {
		return waitForConnection().readDouble();
	}

	@Override
	public boolean readBoolean() throws IOException {
		return waitForConnection().readBoolean();
	}

	@Override
	public void sendInputStream(InputStream is) {
		CommandSender connection = this.connection;
		if (connection != null) {
			connection.sendInputStream(is);
		}
	}

	@Override
	public InputStream readInputStream() throws IOException {
		return waitForConnection().readInputStream();
	}

}
