/*
 * Copyright (c) 2000, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.net;

import java.util.Enumeration;

import ej.annotation.Nullable;

/**
 * This class represents a Network Interface made up of a name, and a list of IP addresses assigned to this interface.
 * It is used to identify the local interface on which a multicast group is joined.
 *
 * Interfaces are normally known by names such as "le0".
 *
 * @since 1.4
 */
public final class NetworkInterface {

	/**
	 * Returns an NetworkInterface object with index set to 0 and name to null. Setting such an interface on a
	 * MulticastSocket will cause the kernel to choose one interface for sending multicast packets.
	 *
	 */
	NetworkInterface() {
	}

	/**
	 * Get the name of this network interface.
	 *
	 * @return the name of this network interface
	 */
	@Nullable
	public String getName() {
		throw new RuntimeException();
	}

	/**
	 * Convenience method to return an Enumeration with all or a subset of the InetAddresses bound to this network
	 * interface.
	 * <p>
	 * If there is a security manager, its {@code checkConnect} method is called for each InetAddress. Only
	 * InetAddresses where the {@code checkConnect} doesn't throw a SecurityException will be returned in the
	 * Enumeration. However, if the caller has the {@link NetPermission}("getNetworkInformation") permission, then all
	 * InetAddresses are returned.
	 *
	 * @return an Enumeration object with all or a subset of the InetAddresses bound to this network interface
	 */
	public Enumeration<InetAddress> getInetAddresses() {
		throw new RuntimeException();
	}

	/**
	 * Get a List of all or a subset of the {@code InterfaceAddresses} of this network interface.
	 * <p>
	 * If there is a security manager, its {@code checkConnect} method is called with the InetAddress for each
	 * InterfaceAddress. Only InterfaceAddresses where the {@code checkConnect} doesn't throw a SecurityException will
	 * be returned in the List.
	 *
	 * @return a {@code List} object with all or a subset of the InterfaceAddresss of this network interface
	 * @since 1.6
	 */
	public java.util.List<InterfaceAddress> getInterfaceAddresses() {
		throw new RuntimeException();
	}

	/**
	 * Returns the index of this network interface. The index is an integer greater or equal to zero, or {@code -1} for
	 * unknown. This is a system specific value and interfaces with the same name can have different indexes on
	 * different machines.
	 *
	 * @return the index of this network interface or {@code -1} if the index is unknown
	 * @see #getByIndex(int)
	 * @since 1.7
	 */
	public int getIndex() {
		throw new RuntimeException();
	}

	/**
	 * Get the display name of this network interface. A display name is a human readable String describing the network
	 * device.
	 *
	 * @return a non-empty string representing the display name of this network interface, or null if no display name is
	 *         available.
	 */
	@Nullable
	public String getDisplayName() {
		throw new RuntimeException();
	}

	/**
	 * Searches for the network interface with the specified name.
	 *
	 * @param name
	 *            The name of the network interface.
	 *
	 * @return A {@code NetworkInterface} with the specified name, or {@code null} if there is no network interface with
	 *         the specified name.
	 *
	 * @throws SocketException
	 *             If an I/O error occurs.
	 *
	 * @throws NullPointerException
	 *             If the specified name is {@code null}.
	 */
	@Nullable
	public static NetworkInterface getByName(String name) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Get a network interface given its index.
	 *
	 * @param index
	 *            an integer, the index of the interface
	 * @return the NetworkInterface obtained from its index, or {@code null} if there is no interface with such an index
	 *         on the system
	 * @throws SocketException
	 *             if an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             if index has a negative value
	 * @see #getIndex()
	 * @since 1.7
	 */
	@Nullable
	public static NetworkInterface getByIndex(int index) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Convenience method to search for a network interface that has the specified Internet Protocol (IP) address bound
	 * to it.
	 * <p>
	 * If the specified IP address is bound to multiple network interfaces it is not defined which network interface is
	 * returned.
	 *
	 * @param addr
	 *            The {@code InetAddress} to search with.
	 *
	 * @return A {@code NetworkInterface} or {@code null} if there is no network interface with the specified IP
	 *         address.
	 *
	 * @throws SocketException
	 *             If an I/O error occurs.
	 *
	 * @throws NullPointerException
	 *             If the specified address is {@code null}.
	 */
	@Nullable
	public static NetworkInterface getByInetAddress(InetAddress addr) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns all the interfaces on this machine. Returns null if no network interfaces could be found on this machine.
	 *
	 * NOTE: can use getNetworkInterfaces()+getInetAddresses() to obtain all IP addresses for this node
	 *
	 * @return an Enumeration of NetworkInterfaces found on this machine
	 * @exception SocketException
	 *                if an I/O error occurs.
	 */
	@Nullable
	public static Enumeration<NetworkInterface> getNetworkInterfaces() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns whether a network interface is up and running.
	 *
	 * @return {@code true} if the interface is up and running.
	 * @exception SocketException
	 *                if an I/O error occurs.
	 * @since 1.6
	 */

	public boolean isUp() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns whether a network interface is a loopback interface.
	 *
	 * @return {@code true} if the interface is a loopback interface.
	 * @exception SocketException
	 *                if an I/O error occurs.
	 * @since 1.6
	 */

	public boolean isLoopback() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns whether a network interface supports multicasting or not.
	 *
	 * @return {@code true} if the interface supports Multicasting.
	 * @exception SocketException
	 *                if an I/O error occurs.
	 * @since 1.6
	 */

	public boolean supportsMulticast() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns the hardware address (usually MAC) of the interface if it has one and if it can be accessed given the
	 * current privileges. If a security manager is set, then the caller must have the permission
	 * {@link NetPermission}("getNetworkInformation").
	 *
	 * @return a byte array containing the address, or {@code null} if the address doesn't exist, is not accessible or a
	 *         security manager is set and the caller does not have the permission
	 *         NetPermission("getNetworkInformation")
	 *
	 * @exception SocketException
	 *                if an I/O error occurs.
	 * @since 1.6
	 */
	@Nullable
	public byte[] getHardwareAddress() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Returns the Maximum Transmission Unit (MTU) of this interface.
	 *
	 * @return the value of the MTU for that interface.
	 * @exception SocketException
	 *                if an I/O error occurs.
	 * @since 1.6
	 */
	public int getMTU() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Compares this object against the specified object. The result is {@code true} if and only if the argument is not
	 * {@code null} and it represents the same NetworkInterface as this object.
	 * <p>
	 * Two instances of {@code NetworkInterface} represent the same NetworkInterface if both name and addrs are the same
	 * for both.
	 *
	 * @param obj
	 *            the object to compare against.
	 * @return {@code true} if the objects are the same; {@code false} otherwise.
	 * @see java.net.InetAddress#getAddress()
	 */
	@Override
	public boolean equals(@Nullable Object obj) {
		throw new RuntimeException();
	}

	@Override
	public int hashCode() {
		throw new RuntimeException();
	}

	@Override
	public String toString() {
		throw new RuntimeException();
	}
}
