/*
 * Java
 *
 * Copyright 2015 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.event.controller;

import ej.microui.event.generator.Pointer;

/**
 * Event handler that manages {@link Pointer} events.
 * 
 * @since 2.0
 */
public interface PointerEventHandler {

	/**
	 * Handles pointer pressed event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerPressed(Pointer pointer, int pointerX, int pointerY, int event);

	/**
	 * Handles pointer released event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerReleased(Pointer pointer, int pointerX, int pointerY, int event);

	/**
	 * Handles pointer moved event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerMoved(Pointer pointer, int pointerX, int pointerY, int event);

	/**
	 * Handles pointer dragged event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerDragged(Pointer pointer, int pointerX, int pointerY, int event);

	/**
	 * Handles pointer entered event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerEntered(Pointer pointer, int pointerX, int pointerY, int event);

	/**
	 * Handles pointer exited event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerExited(Pointer pointer, int pointerX, int pointerY, int event);

	/**
	 * Handles pointer clicked event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerClicked(Pointer pointer, int pointerX, int pointerY, int event);

	/**
	 * Handles pointer double-clicked event.
	 *
	 * @param pointer
	 *            the pointer that generates the event.
	 * @param pointerX
	 *            the current x coordinate of the pointer.
	 * @param pointerY
	 *            the current y coordinate of the pointer.
	 * @param event
	 *            the pointer event to handle.
	 * @return <code>true</code> if the event is consumed, <code>false</code> otherwise.
	 */
	boolean onPointerDoubleClicked(Pointer pointer, int pointerX, int pointerY, int event);

}
