/*
 * Java
 *
 * Copyright 2014-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.websocket;

/**
 * A {@link ReasonForClosure} is a close code (which is a number) and a close reason (which a String that may be empty).
 * <p>
 * As explained in section 7.4. Status Codes, "an endpoint may indicate a reason for closure" when sending a close frame. If it does so, the reason
 * for closure is composed of two parts (section 5.5.1 Close):
 * <ul>
 * <li>'status code' (section 7.4) or 'close code' (section 7.1.5), which is a 2-byte unsigned integer,
 * <li>'close reason' (section 7.1.6), which may be empty.
 * </ul>
 *
 * Both are optional (sections 5.5.1 and 7.4) but it is not possible to have a close reason without a close code (sections 5.5.1 and 7.1.6.). Hence,
 * the payload of the close frame may be:
 * <ul>
 * <li>empty (no close code, no close reason),
 * <li>2-byte long (close code, but no close reason),
 * <li>more than 2-byte long (close code, close reason).
 * </ul>
 * When no close code is given in the closing frame, it is considered to be 1005 (section 7.1.5). When no close reason is given in the closing frame,
 * it is the empty string (section 7.1.6).
 *
 * @see CloseCodes
 *
 *
 *
 */
public final class ReasonForClosure {
	/*
	 * Properly speaking, the 'status code' is the value in the frame, whereas the 'close code' is the code given to the user. Since we may receive no
	 * status code, we still need to determine a code to give to the user (the close code). In this case, the value is 1005.
	 */
	/*
	 * Properly speaking, the 'close reason' (section 7.1.6) or 'reason' (section 5.5.1) designates only the UTF-8 message. Nevertheless, the status
	 * code is often referred to as the "reason for closure" (section 7.1.6).
	 */

	private static final String RFC_EMPTY_STRING = ""; //$NON-NLS-1$
	private int code;
	private String reason;

	/**
	 * @see CloseCodes
	 *
	 * @param code
	 *            the close code
	 * @param reason
	 *            a message that describes why the connection has been closed. It may be empty. If it is null, it is converted to the empty string.
	 */
	public ReasonForClosure(int code, String reason) {
		this.code = code;
		setReason(reason);
	}


	/**
	 * Get the close code.
	 *
	 * @return the close code
	 */
	public int getCloseCode() {
		return code;
	}

	/**
	 * Change the close code.
	 *
	 * @param code
	 *            the new code
	 * @return this
	 */
	public ReasonForClosure setCode(int code) {
		this.code = code;
		return this;
	}

	/**
	 * Get the close reason (may be an empty string).
	 *
	 * @return a message that describes why the connection has been closed
	 */
	public String getCloseReason() {
		return reason;
	}

	/**
	 * Change the close reason.
	 *
	 * @param reason
	 *            the new reason message. It may be empty. If it is null, it is converted to the empty string.
	 * @return this
	 */
	public ReasonForClosure setReason(String reason) {
		if (reason == null) {
			// RFC6455 says "empty string", so null is silently translated to empty string
			reason = RFC_EMPTY_STRING;
		}
		this.reason = reason;
		return this;
	}

	@Override
	public String toString() {
		return reason + '(' + code + ')';
	}
}
