/*
 * Copyright 2013-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.service;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.service.loader.CompositeServiceLoader;
import ej.service.loader.ServiceLoaderHelper;
import ej.service.loader.SystemPropertiesServiceLoader;
import ej.service.registry.SimpleServiceRegistry;

/**
 * A simple dependency injection facility.
 * <p>
 * Gets a service loader that is able to retrieve services implementation.
 * <p>
 * The service loader implementation is retrieved from system properties (see {@link SystemPropertiesServiceLoader}), if
 * none is found, a {@link SystemPropertiesServiceLoader} is used. The implementation may be a
 * {@link CompositeServiceLoader} that searches to resolve the services in a list of service loaders.
 * <p>
 * Usage:
 *
 * <pre>
 * MyInterface instance = ServiceFactory.getServiceLoader().getService(MyInterface.class);
 * </pre>
 *
 * <p>
 * The service registry implementation is retrieved from system properties (see {@link SystemPropertiesServiceLoader}),
 * if none is found, a {@link SimpleServiceRegistry} is used.
 * <p>
 * Usage:
 *
 * <pre>
 * ServiceFactory.getServiceRegistry().register(MyInterface.class, myInstance);
 * </pre>
 */
public class ServiceFactory {

	// The service registry must be initialized before the service loader because of the former can be referenced by the
	// instance of the latter.
	private static final ServiceRegistry SERVICE_REGISTRY = getServiceRegistryInstance();
	private static final ServiceLoader SERVICE_LOADER = getServiceLoaderInstance();

	private ServiceFactory() {
	}

	private static ServiceRegistry getServiceRegistryInstance() {
		ServiceLoader propertiesServiceLoader = new SystemPropertiesServiceLoader();
		ServiceRegistry serviceRegistryCandidate = propertiesServiceLoader.getService(ServiceRegistry.class);
		if (serviceRegistryCandidate != null) {
			return serviceRegistryCandidate;
		} else {
			return new SimpleServiceRegistry();
		}
	}

	private static ServiceLoader getServiceLoaderInstance() {
		ServiceLoader propertiesServiceLoader = new SystemPropertiesServiceLoader();
		ServiceLoader serviceLoaderCandidate = propertiesServiceLoader.getService(ServiceLoader.class);
		if (serviceLoaderCandidate != null) {
			return serviceLoaderCandidate;
		} else {
			return propertiesServiceLoader;
		}
	}

	/**
	 * Gets a service loader unique instance.
	 *
	 * @return a service loader.
	 */
	public static ServiceLoader getServiceLoader() {
		return SERVICE_LOADER;
	}

	/**
	 * Gets a service registry unique instance.
	 *
	 * @return a service registry.
	 */
	public static ServiceRegistry getServiceRegistry() {
		return SERVICE_REGISTRY;
	}

	/**
	 * Gets the instance of the given service.
	 * <p>
	 * It first searches in the service loader, then in the service registry.
	 * <p>
	 * May return <code>null</code> if the service is not defined.
	 *
	 * @param service
	 *            the service.
	 * @param <T>
	 *            the service type.
	 * @return the instance of the given service.
	 * @throws ServiceLoadingException
	 *             if an error occurs while instantiating the service instance.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to retrieve the given service instance.
	 * @see #getServiceLoader()
	 * @see #getServiceRegistry()
	 */
	@Nullable
	public static <T> T getService(Class<T> service) {
		synchronized (service) {
			T result = SERVICE_LOADER.getService(service);
			if (result != null) {
				return result;
			}
			return SERVICE_REGISTRY.getService(service);
		}
	}

	/**
	 * Gets the instance of the given service in the service loader then in the service registry (see
	 * {@link #getService(Class)}.
	 * <p>
	 * If there is no registered implementation, an instance of the default implementation is created, registered in the
	 * service registry and returned.
	 *
	 * @param service
	 *            the service.
	 * @param defaultImplementation
	 *            the default implementation to use.
	 * @param <T>
	 *            the service type.
	 * @return the instance of the given service.
	 * @throws ServiceLoadingException
	 *             if an error occurs while instantiating the service instance.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to retrieve the given service instance.
	 * @see #getServiceLoader()
	 * @see #getServiceRegistry()
	 */
	public static <T> T getService(Class<T> service, Class<? extends T> defaultImplementation) {
		synchronized (service) {
			T result = getService(service);
			if (result != null) {
				return result;
			}
			T instantiatedService = ServiceLoaderHelper.createClassInstance(service, defaultImplementation);
			SERVICE_REGISTRY.register(service, instantiatedService);
			return instantiatedService;
		}
	}

	/**
	 * Gets the instance of the given service.
	 * <p>
	 * If there is already an instance in cache, it is returned otherwise a new one is created.
	 * <p>
	 * May throw a {@link MissingServiceException} if the service is not defined.
	 *
	 * @param service
	 *            the service.
	 * @param <T>
	 *            the service type.
	 * @return the instance of the given service.
	 * @throws ServiceLoadingException
	 *             if an error occurs while instantiating the service instance.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to retrieve the given service instance.
	 * @throws MissingServiceException
	 *             if the service has no registered implementation.
	 * @see ServiceLoader#getService(Class)
	 */
	public static <T> T getRequiredService(Class<T> service) {
		T instance = getService(service);
		if (instance == null) {
			@SuppressWarnings("null") // Class name is not null for sure.
			@NonNull
			String name = service.getName();
			throw new MissingServiceException(name);
		}
		return instance;
	}

}