/*
 * Java
 *
 * Copyright 2019-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.library.iot.rcommand.bluetooth.commands;

import java.io.IOException;

import ej.annotation.Nullable;
import ej.bluetooth.BluetoothAdapter;
import ej.bluetooth.BluetoothScanFilter;
import ej.library.iot.rcommand.bluetooth.Commands;
import ej.rcommand.CommandReader;
import ej.rcommand.CommandSender;
import ej.rcommand.synchronous.Endpoint;

public class StartScanningEndpoint implements Endpoint {

	private static final byte ACTION_NONE = 0;
	private static final byte ACTION_FIELD_EXISTS = 1;
	private static final byte ACTION_FIELD_EQUALS = 2;
	private static final byte ACTION_FIELD_STARTS_WITH = 3;

	private byte filterAction;
	private byte filterType;
	private @Nullable byte[] filterData;

	@Override
	public String getName() {
		return Commands.BLUETOOTH_START_SCANNING;
	}

	@Override
	public void readRequestBody(CommandReader reader) throws IOException {
		this.filterAction = (byte) reader.readInt();
		this.filterType = (byte) reader.readInt();
		this.filterData = reader.readByteArray();
	}

	@Override
	public void writeResponseBody(CommandSender sender) throws IOException {
		BluetoothScanFilter scanFilter = createScanFilter();
		boolean success;
		if (scanFilter == null) {
			success = false;
		} else {
			success = BluetoothAdapter.getAdapter().startScanning(scanFilter);
		}

		if (success) {
			sender.sendInt(Commands.OK);
		} else {
			sender.sendInt(Commands.ERROR);
		}
	}

	private @Nullable BluetoothScanFilter createScanFilter() {
		switch (this.filterAction) {
		case ACTION_NONE:
			return BluetoothScanFilter.noFilter();
		case ACTION_FIELD_EXISTS:
			return BluetoothScanFilter.fieldExists(this.filterType);
		case ACTION_FIELD_EQUALS:
			assert (this.filterData != null);
			return BluetoothScanFilter.fieldEquals(this.filterType, this.filterData);
		case ACTION_FIELD_STARTS_WITH:
			assert (this.filterData != null);
			return BluetoothScanFilter.fieldStartsWith(this.filterType, this.filterData);
		default:
			return null;
		}
	}
}
