/*
 * Java
 *
 * Copyright 2020-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.drawing;

import ej.microui.display.LLUIDisplay;
import ej.microui.display.LLUIPainter;
import ej.microui.display.MicroUIGraphicsContext;
import ej.microui.display.MicroUIImage;
import ej.microui.display.MicroUIImageFormat;

/**
 * This class provides implementations of native methods of ej.drawing library which perform some drawings.
 * <p>
 * The equivalent of embedded structures and enumerations are {@link DrawingFlip} and {@link DrawingCap}. They are used
 * to identify the kind of drawing. Additionally, these methods requires {@link MicroUIImage},
 * {@link MicroUIGraphicsContext} and {@link MicroUIImageFormat} classes.
 * <P>
 * The methods implementations follow the same rules than {@link LLUIPainter} (main important thing is to use the
 * framework object {@link LLUIDisplay#Instance} as monitor to synchronize drawings).
 */
public class LLDWPainter {

	/**
	 * Defines the transformation to apply when drawing an image with flip.
	 */
	public enum DrawingFlip {
		/**
		 * Defines the flip mode "none": no flip must be performed.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 */
		DRAWING_FLIP_NONE(0),

		/**
		 * Defines the flip mode 90°: image must be rotated to 90° anticlockwise.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * B D F
		 * A C E
		 * </pre>
		 */
		DRAWING_FLIP_90(7),

		/**
		 * Defines the flip mode 180°: image must be rotated to 180° anticlockwise.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * F E
		 * D C
		 * B A
		 * </pre>
		 */
		DRAWING_FLIP_180(4),

		/**
		 * Defines the flip mode 270°: image must be rotated to 270° anticlockwise.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * E C A
		 * F D B
		 * </pre>
		 */
		DRAWING_FLIP_270(2),

		/**
		 * Defines the flip mode mirror + "none": only a vertical mirror must be applied.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * B A
		 * D C
		 * F E
		 * </pre>
		 */
		DRAWING_FLIP_MIRROR(3),

		/**
		 * Defines the flip mode mirror + 90°: a vertical mirror must be applied and then the image must be rotated to
		 * 90° anticlockwise.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * A C E
		 * B D F
		 * </pre>
		 *
		 * and then mirrored
		 */
		DRAWING_FLIP_MIRROR_90(1),

		/**
		 * Defines the flip mode mirror + 180°: a vertical mirror must be applied and then the image must be rotated to
		 * 180° anticlockwise.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * E F
		 * C D
		 * A B
		 * </pre>
		 */
		DRAWING_FLIP_MIRROR_180(6),

		/**
		 * Defines the flip mode mirror + 270°: a vertical mirror must be applied and then the image must be rotated to
		 * 270° anticlockwise.
		 *
		 * The image
		 *
		 * <pre>
		 * A B
		 * C D
		 * E F
		 * </pre>
		 *
		 * is drawn:
		 *
		 * <pre>
		 * F D B
		 * E C A
		 * </pre>
		 */
		DRAWING_FLIP_MIRROR_270(5);

		/**
		 * ej.drawing native internal flip representation.
		 */
		public final int val;

		private DrawingFlip(int val) {
			this.val = val;
		}
	}

	/**
	 * Defines the start and end caps to draw when rendering an anti-aliased drawing.
	 */
	public enum DrawingCap {
		/**
		 * No specific cap is drawn.
		 */
		DRAWING_ENDOFLINE_NONE(0),

		/**
		 * Cap is represented by a semi circle.
		 */
		DRAWING_ENDOFLINE_ROUNDED(1),

		/**
		 * Cap is perpendicular to the line.
		 */
		DRAWING_ENDOFLINE_PERPENDICULAR(2);

		/**
		 * ej.drawing native internal cap representation.
		 */
		public final int val;

		private DrawingCap(int val) {
			this.val = val;
		}
	}

	private LLDWPainter() {
		// forbidden constructor: there is no instance of LLDWPainter
	}

	/**
	 * ej.drawing native "drawThickFadedPoint". Draws a thick point with fade at given position.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the point X coordinate.
	 * @param y
	 *            the point Y coordinate.
	 * @param thickness
	 *            the point thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	public static void drawThickFadedPoint(byte[] target, int x, int y, int thickness, int fade) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && ((thickness > 0) || (fade > 0))) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickFadedPoint(gc, x, y, thickness, fade);
			}
		}
	}

	/**
	 * ej.drawing native "drawThickFadedLine". Draws a thick line with fade between given points.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x1
	 *            the x coordinate of the start of the line
	 * @param y1
	 *            the y coordinate of the start of the line
	 * @param x2
	 *            the x coordinate of the end of the line
	 * @param y2
	 *            the y coordinate of the end of the line
	 * @param thickness
	 *            the line thickness.
	 * @param fade
	 *            the fade to apply.
	 * @param start
	 *            cap representation of start of shape
	 * @param end
	 *            cap representation of end of shape
	 */
	public static void drawThickFadedLine(byte[] target, int x1, int y1, int x2, int y2, int thickness, int fade,
			int start, int end) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && ((thickness > 0) || (fade > 0))) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickFadedLine(gc, x1, y1, x2, y2, thickness, fade,
						mapCap(start), mapCap(end));
			}
		}
	}

	/**
	 * ej.drawing native "drawThickFadedCircle". Draws a thick circle with fade covering the square specified by its
	 * diameter. If diameter is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the circle is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param thickness
	 *            the circle thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	public static void drawThickFadedCircle(byte[] target, int x, int y, int diameter, int thickness, int fade) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && (diameter > 0) && ((thickness > 0) || (fade > 0))) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickFadedCircle(gc, x, y, diameter, thickness, fade);
			}
		}
	}

	/**
	 * ej.drawing native "drawThickFadedCircleArc". Draws a thick circle with fade arc covering the specified square.
	 * The arc is drawn from startAngle up to arcAngle degrees.
	 * <p>
	 * The center of the arc is defined as the center of the square whose origin is at (x,y) (upper-left corner) and
	 * whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If diameter is negative or zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square. Thus for a non squared square angles are
	 * skewed along either height or width.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the arc is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 * @param thickness
	 *            the arc thickness.
	 * @param fade
	 *            the fade to apply.
	 * @param start
	 *            cap representation of start of shape
	 * @param end
	 *            cap representation of end of shape
	 */
	public static void drawThickFadedCircleArc(byte[] target, int x, int y, int diameter, float startAngle,
			float arcAngle, int thickness, int fade, int start, int end) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && (diameter > 0) && (arcAngle != 0) && ((thickness > 0) || (fade > 0))) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickFadedCircleArc(gc, x, y, diameter, startAngle, arcAngle,
						thickness, fade, mapCap(start), mapCap(end));
			}
		}
	}

	/**
	 * ej.drawing native "drawAAEllipse". Draws a thick ellipse with fade covering the specified rectangle.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at (x,y) (upper-left corner)
	 * and whose dimension is given by width and height.
	 * <p>
	 * If either width or height is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param width
	 *            the width of the ellipse to draw
	 * @param height
	 *            the height of the ellipse to draw
	 * @param thickness
	 *            the ellipse thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	public static void drawThickFadedEllipse(byte[] target, int x, int y, int width, int height, int thickness,
			int fade) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && (width > 0) && (height > 0) && ((thickness > 0) || (fade > 0))) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickFadedEllipse(gc, x, y, width, height, thickness, fade);
			}
		}
	}

	/**
	 * ej.drawing native "drawThickLine". Draws a thick line between given points.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x1
	 *            the x coordinate of the start of the line
	 * @param y1
	 *            the y coordinate of the start of the line
	 * @param x2
	 *            the x coordinate of the end of the line
	 * @param y2
	 *            the y coordinate of the end of the line
	 * @param thickness
	 *            the line thickness.
	 */
	public static void drawThickLine(byte[] target, int x1, int y1, int x2, int y2, int thickness) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && (thickness > 0)) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickLine(gc, x1, y1, x2, y2, thickness);
			}
		}
	}

	/**
	 * ej.drawing native "drawThickEllipse". Draws a thick ellipse covering the specified rectangle.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at (x,y) (upper-left corner)
	 * and whose dimension is given by width and height.
	 * <p>
	 * If either width or height is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the circle is drawn
	 * @param width
	 *            the width of the ellipse to draw
	 * @param height
	 *            the height of the ellipse to draw
	 * @param thickness
	 *            the circle thickness.
	 */
	public static void drawThickEllipse(byte[] target, int x, int y, int width, int height, int thickness) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && (width > 0) && (height > 0) && (thickness > 0)) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickEllipse(gc, x, y, width, height, thickness);
			}
		}
	}

	/**
	 * ej.drawing native "drawThickCircle". Draws a thick ellipse covering the square specified by its diameter.
	 * <p>
	 * If diameter is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the circle is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param thickness
	 *            the circle thickness.
	 */
	public static void drawThickCircle(byte[] target, int x, int y, int diameter, int thickness) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && (diameter > 0) && (thickness > 0)) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickCircle(gc, x, y, diameter, thickness);
			}
		}
	}

	/**
	 * ej.drawing native "drawThickCircleArc". Draws a thick arc covering the square specified by its diameter.
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If diameter is negative, nothing is drawn.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the arc is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 * @param thickness
	 *            the arc thickness.
	 */
	public static void drawThickCircleArc(byte[] target, int x, int y, int diameter, float startAngle, float arcAngle,
			int thickness) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && (diameter > 0) && (arcAngle != 0) && (thickness > 0)) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawThickCircleArc(gc, x, y, diameter, startAngle, arcAngle,
						thickness);
			}
		}
	}

	/**
	 * ej.drawing native "drawFlippedImage". Draws an image applying a flip (0, 90, 180 or 270 degrees with or without
	 * mirror).
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param source
	 *            image source: a byte array to map on a {@link MicroUIImage}
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to draw.
	 * @param regionY
	 *            the y coordinate of the upper-left corner of the region to draw.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param flip
	 *            the flip to apply.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	public static void drawFlippedImage(byte[] target, byte[] source, int regionX, int regionY, int width, int height,
			int x, int y, int flip, int alpha) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIImage image = graphicsEngine.mapMicroUIImage(source);
			if (gc.requestDrawing() && alpha > 0 && !image.isClosed()) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawFlippedImage(gc, image, regionX, regionY, width, height, x, y,
						mapFlip(flip), alpha);
			}
		}
	}

	/**
	 * ej.drawing native "drawRotatedImageNearestNeighbor". Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm but its rendering is faster.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param source
	 *            image source: a byte array to map on a {@link MicroUIImage}
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	public static void drawRotatedImageNearestNeighbor(byte[] target, byte[] source, int x, int y, int rotationX,
			int rotationY, float angle, int alpha) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIImage image = graphicsEngine.mapMicroUIImage(source);
			if (gc.requestDrawing() && alpha > 0 && !image.isClosed()) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawRotatedImageNearestNeighbor(gc, image, x, y, rotationX,
						rotationY, angle, alpha);
			}
		}
	}

	/**
	 * ej.drawing native "drawRotatedImageBilinear". Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param source
	 *            image source: a byte array to map on a {@link MicroUIImage}
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	public static void drawRotatedImageBilinear(byte[] target, byte[] source, int x, int y, int rotationX,
			int rotationY, float angle, int alpha) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIImage image = graphicsEngine.mapMicroUIImage(source);
			if (gc.requestDrawing() && alpha > 0 && !image.isClosed()) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawRotatedImageBilinear(gc, image, x, y, rotationX, rotationY,
						angle, alpha);
			}
		}
	}

	/**
	 * ej.drawing native "drawScaledImageNearestNeighbor". Draws an image applying a scaling.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm but its rendering is faster.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param source
	 *            image source: a byte array to map on a {@link MicroUIImage}
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	public static void drawScaledImageNearestNeighbor(byte[] target, byte[] source, int x, int y, float factorX,
			float factorY, int alpha) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIImage image = graphicsEngine.mapMicroUIImage(source);
			if (gc.requestDrawing() && alpha > 0 && !image.isClosed()) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawScaledImageNearestNeighbor(gc, image, x, y, factorX, factorY,
						alpha);
			}
		}
	}

	/**
	 * ej.drawing native "drawScaledImageBilinear". Draws an image applying a scaling.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param source
	 *            image source: a byte array to map on a {@link MicroUIImage}
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	public static void drawScaledImageBilinear(byte[] target, byte[] source, int x, int y, float factorX, float factorY,
			int alpha) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIImage image = graphicsEngine.mapMicroUIImage(source);
			if (gc.requestDrawing() && alpha > 0 && !image.isClosed()) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawScaledImageBilinear(gc, image, x, y, factorX, factorY, alpha);
			}
		}
	}

	/**
	 * Gets the {@link DrawingCap} which matches with the given value. This value is the ej.drawing native parameter.
	 *
	 * @param value
	 *            the ej.drawing native parameter.
	 * @return the equivalent in {@link DrawingCap} format.
	 * @throws IllegalArgumentException
	 *             if value does not match a {@link DrawingCap} value.
	 */
	private static DrawingCap mapCap(int value) {
		switch (value) {
		case 0:
			return DrawingCap.DRAWING_ENDOFLINE_NONE;
		case 1:
			return DrawingCap.DRAWING_ENDOFLINE_ROUNDED;
		case 2:
			return DrawingCap.DRAWING_ENDOFLINE_PERPENDICULAR;
		default:
			throw new IllegalArgumentException();
		}
	}

	/**
	 * Gets the {@link DrawingFlip} which matches with the given value. This value is the ej.drawing native parameter.
	 *
	 * @param value
	 *            the ej.drawing native parameter.
	 * @return the equivalent in {@link DrawingFlip} format.
	 * @throws IllegalArgumentException
	 *             if value does not match a {@link DrawingFlip} value.
	 */
	private static DrawingFlip mapFlip(int value) {
		switch (value) {
		case 0:
			return DrawingFlip.DRAWING_FLIP_NONE;
		case 1:
			return DrawingFlip.DRAWING_FLIP_MIRROR_90;
		case 2:
			return DrawingFlip.DRAWING_FLIP_270;
		case 3:
			return DrawingFlip.DRAWING_FLIP_MIRROR;
		case 4:
			return DrawingFlip.DRAWING_FLIP_180;
		case 5:
			return DrawingFlip.DRAWING_FLIP_MIRROR_270;
		case 6:
			return DrawingFlip.DRAWING_FLIP_MIRROR_180;
		case 7:
			return DrawingFlip.DRAWING_FLIP_90;
		default:
			throw new IllegalArgumentException();
		}
	}
}
