/*
 * Copyright (C) 2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.microej.http.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

import ej.bon.Constants;

/**
 * Factory class for BufferStream instantiation.
 */
public class HttpBufferedStreamFactory {

	private static final int MIN_RX_BUFFER_SIZE = 10;
	private static final String BUFFER_SIZE = ".buffer.size";
	private static final String JAVA_NET_HTTP_URL_CONNECTION = "java.net.HttpURLConnection.";
	private static final String RX_BUFFER_SIZE_CONSTANTE_NAME = JAVA_NET_HTTP_URL_CONNECTION + "rx" + BUFFER_SIZE;
	private static final String TX_BUFFER_SIZE_CONSTANTE_NAME = JAVA_NET_HTTP_URL_CONNECTION + "tx" + BUFFER_SIZE;

	private HttpBufferedStreamFactory() {
		// forbid instantiation.
	}

	/**
	 * Generates a new BufferInputStream with the buffered size defined by {@value #RX_BUFFER_SIZE_CONSTANTE_NAME}. The
	 * minimum buffer size is {@value #MIN_RX_BUFFER_SIZE}.
	 *
	 * @param inputStream
	 *            the input stream.
	 * @return a BufferedInputStream with the buffer size defined by {@value #RX_BUFFER_SIZE_CONSTANTE_NAME}.
	 */
	public static InputStream newInputStream(InputStream inputStream) {
		// HttpClient#parseHTTPHeader uses mark and reset with a size of 10 the mark is currently not supported by
		// InputStream (and SocketInputStream). So we need to use a BufferedInputStream with at least 10 bytes.
		return new BufferedInputStream(inputStream,
				Math.max(Constants.getInt(RX_BUFFER_SIZE_CONSTANTE_NAME), MIN_RX_BUFFER_SIZE));
	}

	/**
	 * Generates a new BufferOutputStream with the buffered size defined by {@value #TX_BUFFER_SIZE_CONSTANTE_NAME}.
	 *
	 * @param outputStream
	 *            the output stream.
	 * @return a BufferedOutputStream with the buffer size defined by {@value #TX_BUFFER_SIZE_CONSTANTE_NAME} if it is
	 *         more than 0, outputStream otherwise.
	 */
	public static OutputStream newOutputStream(OutputStream outputStream) {
		int bufferSize = Constants.getInt(TX_BUFFER_SIZE_CONSTANTE_NAME);
		if (bufferSize > 0) {
			return new BufferedOutputStream(outputStream, bufferSize);
		} else {
			return outputStream;
		}
	}
}
