/*
 * Java
 *
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.rcommand.serversocket;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.Socket;

import ej.rcommand.impl.StreamRemoteConnection;

/**
 * The connection with a remote socket that uses buffered streams.
 *
 * The size of the buffers used is 8192 by default and can be modified by
 * setting the property <code>rcommand.serversocket.buffer.size</code>.
 */
public class SocketRemoteConnection extends StreamRemoteConnection {

	private static final String BUFFER_SIZE_PROPERTY = "rcommand.serversocket.buffer.size";
	private static final int DEFAULT_BUFFER_SIZE = 8192;

	private final RemoteCommandServer adminServer;
	private final Socket socket;
	private boolean closed;

	/**
	 * @param socket
	 * @param adminServer
	 * @throws IOException
	 */
	public SocketRemoteConnection(Socket socket, RemoteCommandServer adminServer) throws IOException {
		super(new BufferedInputStream(socket.getInputStream(), getBufferSize()),
				new BufferedOutputStream(socket.getOutputStream(), getBufferSize()));
		this.socket = socket;
		this.adminServer = adminServer;
	}

	@Override
	public void close() {
		if (!this.closed) {
			this.closed = true;
			this.adminServer.releaseConnection();
			super.close();
			try {
				this.socket.close();
			} catch (IOException e) {
				// do nothing
			}
		}
	}

	private static int getBufferSize() {
		return Integer.getInteger(BUFFER_SIZE_PROPERTY, DEFAULT_BUFFER_SIZE).intValue();
	}

}
