/*
 * Java
 *
 * Copyright 2018-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.services.sps;

import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothConnection;
import ej.bluetooth.BluetoothDescriptor;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.BluetoothStatus;
import ej.bluetooth.listeners.impl.DefaultRemoteServiceListener;
import ej.bluetooth.util.AttributeNotFoundException;
import ej.bluetooth.util.DescriptorHelper;
import ej.bluetooth.util.ServiceHelper;

/**
 * The <code>SerialPortClient</code> class represents a serial port client.
 */
public abstract class SerialPortClient extends DefaultRemoteServiceListener {

	private final BluetoothConnection connection;

	private final BluetoothService service;
	private final BluetoothCharacteristic tx;
	private final BluetoothCharacteristic rx;
	private final BluetoothDescriptor txCcc;

	/**
	 * Creates a serial port client, using the serial port service provided by a remote device and the connection to
	 * this device.
	 *
	 * @param connection
	 *            the connection to the device.
	 * @param service
	 *            the serial port service provided by the device.
	 * @throws AttributeNotFoundException
	 *             if one of the mandatory attributes of the service is missing.
	 */
	public SerialPortClient(BluetoothConnection connection, BluetoothService service)
			throws AttributeNotFoundException {
		this.connection = connection;

		this.service = service;
		this.tx = ServiceHelper.getCharacteristic(service, SerialPortConstants.TX_UUID);
		this.rx = ServiceHelper.getCharacteristic(service, SerialPortConstants.RX_UUID);
		this.txCcc = ServiceHelper.getDescriptor(this.tx, DescriptorHelper.CCC_UUID);

		service.setRemoteListener(this);

		byte[] cccValue = DescriptorHelper.createCccValue(true, false);
		connection.sendWriteRequest(this.txCcc, cccValue);
	}

	/**
	 * Closes this serial port client.
	 */
	public void close() {
		this.service.setRemoteListener(new DefaultRemoteServiceListener());
	}

	/**
	 * Sends the given data to the server. The {@link #onDataSent(boolean)} methods is called once the data has been
	 * sent to the server.
	 *
	 * @param data
	 *            the data to send.
	 */
	public void sendData(byte[] data) {
		this.connection.sendWriteRequest(this.rx, data);
	}

	@Override
	public void onWriteCompleted(BluetoothConnection connection, BluetoothAttribute attribute, byte status) {
		if (attribute == this.rx) {
			onDataSent(status == BluetoothStatus.OK);
		}
	}

	@Override
	public void onNotificationReceived(BluetoothConnection connection, BluetoothCharacteristic characteristic,
			byte[] value) {
		if (characteristic == this.tx) {
			onDataReceived(value);
		}
	}

	/**
	 * Called when data has been received from the server.
	 *
	 * @param data
	 *            the data received from the server.
	 */
	protected abstract void onDataReceived(byte[] data);

	/**
	 * Called when data has been sent to the server.
	 *
	 * @param success
	 *            true if the the data was sent successfully, false otherwise.
	 */
	protected abstract void onDataSent(boolean success);
}
