/*
 * Protocol Buffers - Google's data interchange format
 * Copyright 2008 Google Inc.  All rights reserved.
 * https: *developers.google.com/protocol-buffers/
 * Copyright 2020 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.google.protobuf;

import java.util.Arrays;
import java.util.Collection;
import java.util.RandomAccess;

import com.google.protobuf.Internal.BooleanList;

/**
 * An implementation of {@link BooleanList} on top of a primitive array.
 *
 * @author dweis@google.com (Daniel Weis)
 */
final class BooleanArrayList extends AbstractProtobufList<Boolean> implements BooleanList, RandomAccess {

	private static final BooleanArrayList EMPTY_LIST = new BooleanArrayList();
	static {
		EMPTY_LIST.makeImmutable();
	}

	public static BooleanArrayList emptyList() {
		return EMPTY_LIST;
	}

	/**
	 * The backing store for the list.
	 */
	private boolean[] array;

	/**
	 * The size of the list distinct from the length of the array. That is, it is the number of elements set in the
	 * list.
	 */
	private int size;

	/**
	 * Constructs a new mutable {@code BooleanArrayList} with default capacity.
	 */
	BooleanArrayList() {
		this(new boolean[DEFAULT_CAPACITY], 0);
	}

	/**
	 * Constructs a new mutable {@code BooleanArrayList} containing the same elements as {@code other}.
	 */
	private BooleanArrayList(boolean[] other, int size) {
		this.array = other;
		this.size = size;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof BooleanArrayList)) {
			return super.equals(o);
		}
		BooleanArrayList other = (BooleanArrayList) o;
		if (this.size != other.size) {
			return false;
		}

		final boolean[] arr = other.array;
		for (int i = 0; i < this.size; i++) {
			if (this.array[i] != arr[i]) {
				return false;
			}
		}

		return true;
	}

	@Override
	public int hashCode() {
		int result = 1;
		for (int i = 0; i < this.size; i++) {
			result = (31 * result) + Internal.hashBoolean(this.array[i]);
		}
		return result;
	}

	@Override
	public BooleanList mutableCopyWithCapacity(int capacity) {
		if (capacity < this.size) {
			throw new IllegalArgumentException();
		}
		return new BooleanArrayList(Arrays.copyOf(this.array, capacity), this.size);
	}

	@Override
	public Boolean get(int index) {
		return Boolean.valueOf(getBoolean(index));
	}

	@Override
	public boolean getBoolean(int index) {
		ensureIndexInRange(index);
		return this.array[index];
	}

	@Override
	public int size() {
		return this.size;
	}

	@Override
	public Boolean set(int index, Boolean element) {
		return Boolean.valueOf(setBoolean(index, element.booleanValue()));
	}

	@Override
	public boolean setBoolean(int index, boolean element) {
		ensureIsMutable();
		ensureIndexInRange(index);
		boolean previousValue = this.array[index];
		this.array[index] = element;
		return previousValue;
	}

	@Override
	public void add(int index, Boolean element) {
		addBoolean(index, element.booleanValue());
	}

	/**
	 * Like {@link #add(Boolean)} but more efficient in that it doesn't box the element.
	 */
	@Override
	public void addBoolean(boolean element) {
		addBoolean(this.size, element);
	}

	/**
	 * Like {@link #add(int, Boolean)} but more efficient in that it doesn't box the element.
	 */
	private void addBoolean(int index, boolean element) {
		ensureIsMutable();
		if (index < 0 || index > this.size) {
			throw new IndexOutOfBoundsException(makeOutOfBoundsExceptionMessage(index));
		}

		if (this.size < this.array.length) {
			// Shift everything over to make room
			System.arraycopy(this.array, index, this.array, index + 1, this.size - index);
		} else {
			// Resize to 1.5x the size
			int length = ((this.size * 3) / 2) + 1;
			boolean[] newArray = new boolean[length];

			// Copy the first part directly
			System.arraycopy(this.array, 0, newArray, 0, index);

			// Copy the rest shifted over by one to make room
			System.arraycopy(this.array, index, newArray, index + 1, this.size - index);
			this.array = newArray;
		}

		this.array[index] = element;
		this.size++;
		this.modCount++;
	}

	@Override
	public boolean addAll(Collection<? extends Boolean> collection) {
		ensureIsMutable();

		if (collection == null) {
			throw new NullPointerException();
		}

		// We specialize when adding another BooleanArrayList to avoid boxing elements.
		if (!(collection instanceof BooleanArrayList)) {
			return super.addAll(collection);
		}

		BooleanArrayList list = (BooleanArrayList) collection;
		if (list.size == 0) {
			return false;
		}

		int overflow = Integer.MAX_VALUE - this.size;
		if (overflow < list.size) {
			// We can't actually represent a list this large.
			throw new OutOfMemoryError();
		}

		int newSize = this.size + list.size;
		if (newSize > this.array.length) {
			this.array = Arrays.copyOf(this.array, newSize);
		}

		System.arraycopy(list.array, 0, this.array, this.size, list.size);
		this.size = newSize;
		this.modCount++;
		return true;
	}

	@Override
	public boolean remove(Object o) {
		ensureIsMutable();
		for (int i = 0; i < this.size; i++) {
			if (o.equals(Boolean.valueOf(this.array[i]))) {
				System.arraycopy(this.array, i + 1, this.array, i, this.size - i);
				this.size--;
				this.modCount++;
				return true;
			}
		}
		return false;
	}

	@Override
	public Boolean remove(int index) {
		ensureIsMutable();
		ensureIndexInRange(index);
		boolean value = this.array[index];
		if (index < this.size - 1) {
			System.arraycopy(this.array, index + 1, this.array, index, this.size - index - 1);
		}
		this.size--;
		this.modCount++;
		return value;
	}

	/**
	 * Ensures that the provided {@code index} is within the range of {@code [0, size]}. Throws an
	 * {@link IndexOutOfBoundsException} if it is not.
	 *
	 * @param index
	 *            the index to verify is in range
	 */
	private void ensureIndexInRange(int index) {
		if (index < 0 || index >= this.size) {
			throw new IndexOutOfBoundsException(makeOutOfBoundsExceptionMessage(index));
		}
	}

	private String makeOutOfBoundsExceptionMessage(int index) {
		return "Index:" + index + ", Size:" + this.size; //$NON-NLS-1$ //$NON-NLS-2$
	}
}
