/*
 * Copyright 2016-2019 MicroEJ Corp. - EDC compliance and optimizations.
 * Use is subject to license terms.
 */
package org.junit.internal;

//<removed for EDC compliance>import java.lang.reflect.Array;
import java.util.Arrays;

import org.junit.Assert;

/**
 * Defines criteria for finding two items "equal enough". Concrete subclasses may demand exact equality, or, for
 * example, equality within a given delta.
 */
public abstract class ComparisonCriteria {
	/**
	 * Asserts that two arrays are equal, according to the criteria defined by the concrete subclass. If they are not,
	 * an {@link AssertionError} is thrown with the given message. If <code>expecteds</code> and <code>actuals</code>
	 * are <code>null</code>, they are considered equal.
	 *
	 * @param message
	 *            the identifying message for the {@link AssertionError} ( <code>null</code> okay)
	 * @param expecteds
	 *            Object array or array of arrays (multi-dimensional array) with expected values.
	 * @param actuals
	 *            Object array or array of arrays (multi-dimensional array) with actual values
	 */
	public void arrayEquals(String message, Object expecteds, Object actuals) throws ArrayComparisonFailure {
		if (expecteds == actuals || Arrays.deepEquals(new Object[] { expecteds }, new Object[] { actuals })) {
			// The reflection-based loop below is potentially very slow, especially for primitive
			// arrays. The deepEquals check allows us to circumvent it in the usual case where
			// the arrays are exactly equal.
			return;
		}
		String header = message == null ? "" : message + ": ";

		int expectedsLength = assertArraysAreSameLength(expecteds, actuals, header);

		for (int i = 0; i < expectedsLength; i++) {
			// <edited for EDC compliance>
			Object expected = _Array_get(expecteds, i);
			Object actual = _Array_get(actuals, i);

			if (isArray(expected) && isArray(actual)) {
				try {
					arrayEquals(message, expected, actual);
				} catch (ArrayComparisonFailure e) {
					e.addDimension(i);
					throw e;
				}
			} else {
				try {
					assertElementsEqual(expected, actual);
				} catch (AssertionError e) {
					throw new ArrayComparisonFailure(header, e, i);
				}
			}
		}
	}

	private boolean isArray(Object expected) {
		return expected != null && expected.getClass().isArray();
	}

	private int assertArraysAreSameLength(Object expecteds, Object actuals, String header) {
		if (expecteds == null) {
			Assert.fail(header + "expected array was null");
		}
		if (actuals == null) {
			Assert.fail(header + "actual array was null");
		}
		// <removed for EDC compliance>
		int actualsLength = _Array_getLength(actuals);
		int expectedsLength = _Array_getLength(expecteds);
		if (actualsLength != expectedsLength) {
			Assert.fail(header + "array lengths differed, expected.length=" + expectedsLength + " actual.length="
					+ actualsLength);
		}
		return expectedsLength;
	}

	// <added for EDC compliance>
	private int _Array_getLength(Object array) {
		if (array instanceof Object[]) {
			return ((Object[]) array).length;
		} else if (array instanceof byte[]) {
			return ((byte[]) array).length;
		} else if (array instanceof short[]) {
			return ((short[]) array).length;
		} else if (array instanceof int[]) {
			return ((int[]) array).length;
		} else if (array instanceof long[]) {
			return ((long[]) array).length;
		} else if (array instanceof char[]) {
			return ((char[]) array).length;
		} else if (array instanceof float[]) {
			return ((float[]) array).length;
		} else if (array instanceof double[]) {
			return ((double[]) array).length;
		} else if (array instanceof boolean[]) {
			return ((boolean[]) array).length;
		}
		throw new AssertionError("Not an array");
	}

	private Object _Array_get(Object array, int i) {
		if (array instanceof Object[]) {
			return ((Object[]) array)[i];
		} else if (array instanceof byte[]) {
			return new Byte(((byte[]) array)[i]);
		} else if (array instanceof short[]) {
			return new Short(((short[]) array)[i]);
		} else if (array instanceof int[]) {
			return new Integer(((int[]) array)[i]);
		} else if (array instanceof long[]) {
			return new Long(((long[]) array)[i]);
		} else if (array instanceof char[]) {
			return new Character(((char[]) array)[i]);
		} else if (array instanceof float[]) {
			return new Float(((float[]) array)[i]);
		} else if (array instanceof double[]) {
			return new Double(((double[]) array)[i]);
		} else if (array instanceof boolean[]) {
			return new Boolean(((boolean[]) array)[i]);
		}
		throw new AssertionError("Not an array");
	}

	protected abstract void assertElementsEqual(Object expected, Object actual);
}
