/*
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.storage.fs;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;

/**
 * A random access file output stream is an output stream for writing data to a <code>RandomAccessFile</code>.
 */
/* package */ class RandomAccessFileOutputStream extends OutputStream {

	private final RandomAccessFile file;

	/**
	 * Creates a random access file output stream to write to the file represented by the specified <code>File</code>
	 * object.
	 *
	 * @param file
	 *            the file to be opened for writing.
	 * @param position
	 *            the position to write at.
	 * @exception IOException
	 *                if an I/O error occurs.
	 */
	public RandomAccessFileOutputStream(File file, long position) throws IOException {
		this.file = new RandomAccessFile(file, "rw"); //$NON-NLS-1$
		this.file.seek(position);
	}

	@Override
	public void close() throws IOException {
		this.file.close();
	}

	@Override
	public void flush() throws IOException {
		// do nothing: writing automatically flushes
	}

	@Override
	public void write(byte[] b) throws IOException {
		this.file.write(b);
	}

	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		this.file.write(b, off, len);
	}

	@Override
	public void write(int b) throws IOException {
		this.file.writeByte(b);
	}
}
