/*
 * Java
 *
 * Copyright 2015-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.kf.util;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import ej.kf.Converter;
import ej.kf.Feature;
import ej.kf.Kernel;

/**
 * A {@link Converter} for {@link Map}. This converter creates a clone of the original {@link Map}, and convert is
 * called for keys and values.
 *
 * @param <K>
 *            the type of keys maintained by this map
 * @param <V>
 *            the type of mapped values
 */
public class MapConverter<K, V> implements Converter<Map<K, V>> {

	@Override
	@SuppressWarnings({ "unchecked" })
	public Class<Map<K, V>> getType() {
		// this construct is mandatory because of type erasure for class litterals
		return (Class<Map<K, V>>) (Object) Map.class;
	}

	@Override
	public Map<K, V> convert(Map<K, V> map, Feature targetOwner) throws IllegalAccessError {
		final int size = map.size();
		// clone map and strings
		RunnableWithResult<Map<K, V>> runnable = new RunnableWithResult<Map<K, V>>() {
			@Override
			public Map<K, V> runWithResult() {
				return new HashMap<>(size);
			}
		};
		Kernel.runUnderContext(targetOwner, runnable);
		Map<K, V> result = runnable.getResult();
		Set<K> keySet = map.keySet();
		for (K o : keySet) {
			V value = map.get(o);
			// don't clone if value is null
			value = (value == null) ? null : KernelSupport.clone(value, targetOwner);
			result.put(KernelSupport.clone(o, targetOwner), value);
		}
		return result;
	}

}
