/*
 * Java
 *
 * Copyright 2017-2018 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.kf.util;

import java.util.ArrayList;
import java.util.Collection;

import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;

/**
 * Implementation of a {@link Collection} with multi-context support.
 * <p>
 * This class can only be embedded in Kernel code. It cannot be embedded into Feature code.
 * <p>
 * This collection does not allow to pass <code>null</code> elements.
 * <p>
 * The semantic is the following:
 * <ul>
 * <li>Each context has its dedicated list, acting as an {@link ArrayList}.</li>
 * <li>A Feature can add Feature or Kernel objects to its list.</li>
 * <li>The Kernel can only add Kernel objects to its list. However it has the full control and visibility on all the
 * elements of the collection. For example, the call of the {@link #clear()} method clears the whole collection. The
 * call to {@link #iterator()} method return an iterator on all objects of the collection...</li>
 * </ul>
 * <p>
 *
 * @see KFList
 * @see KFList2
 */
public abstract class AbstractKFList<T> implements Collection<T>, FeatureStateListener {

	@Override
	public final synchronized void stateChanged(Feature feature, State previousState) {
		// Remove elements added to the Feature (if any).
		// There is no Feature element into the Kernel list (see KFList semantic specification)
		if (feature.getState() == State.STOPPED) {
			removeAllElementsOwnedBy(feature);
		}
	}

	@Override
	public synchronized boolean addAll(Collection<? extends T> c) {
		boolean result = false;
		for (T e : c) {
			result |= add(e);
		}
		return result;
	}

	@Override
	public synchronized boolean containsAll(Collection<?> c) {
		for (Object e : c) {
			if (!contains(e)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public synchronized boolean removeAll(Collection<?> c) {
		boolean result = false;
		for (Object e : c) {
			result |= remove(e);
		}
		return result;
	}

	@Override
	public synchronized boolean retainAll(Collection<?> c) {
		throw new UnsupportedOperationException();
	}

	@Override
	public synchronized Object[] toArray() {
		ArrayList<T> allList = new ArrayList<T>();
		toArray(allList);
		return allList.toArray();
	}

	@Override
	public synchronized <E> E[] toArray(E[] a) {
		ArrayList<T> allList = new ArrayList<T>();
		toArray(allList);
		allList.toArray(a);
		return a;
	}

	/**
	 * Removed all elements owned by the given {@link Feature}.
	 *
	 * @param feature
	 *            the owner of the elements to be removed
	 */
	protected abstract void removeAllElementsOwnedBy(Feature feature);

	/**
	 * Add all the elements of the collection to the given {@link ArrayList}.
	 *
	 * @param allList
	 *            the list where all elements are added.
	 */
	protected abstract void toArray(ArrayList<T> allList);
}
