/*
 * Java
 *
 * Copyright 2021-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.hoka.session;

import java.util.Set;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;

/**
 * Server side Session.
 */
public class Session {

	private final String id;
	private long expiration;
	private final PackedMap<String, String> attributes;

	/**
	 * Constructs a {@link Session}.
	 *
	 * @param id
	 *            the identifier of the session.
	 * @param expiration
	 *            the expiration date of the session.
	 */
	public Session(String id, long expiration) {
		this.id = id;
		this.expiration = expiration;
		this.attributes = new PackedMap<>();
	}

	/**
	 * Gets the id.
	 *
	 * @return the id.
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * Gets the expiration.
	 *
	 * @return the expiration.
	 */
	public long getExpiration() {
		return this.expiration;
	}

	/**
	 * Checks whether the session has expired or not.
	 *
	 * @return {@code true} if the session has expired, {@code false} otherwise.
	 */
	public boolean hasExpired() {
		return this.expiration < System.currentTimeMillis();
	}

	/**
	 * Set a session attribute.
	 *
	 * @param name
	 *            attribute name
	 * @param value
	 *            attribute value
	 */
	public void setAttribute(String name, String value) {
		this.attributes.put(name, value);
	}

	/**
	 * Get a session attribute.
	 *
	 * @param name
	 *            attribute name
	 * @return return attribute mapped to this name if it exists, null otherwise
	 */
	@Nullable
	public String getAttribute(String name) {
		return this.attributes.get(name);
	}

	/**
	 * Remove an attribute from this session.
	 *
	 * @param name
	 *            attribute name
	 * @return the previous value associated with name, or null if there was no mapping for name.
	 */
	@Nullable
	public String removeAttribute(String name) {
		return this.attributes.remove(name);
	}

	/**
	 * Gets the set of the session attributes names.
	 *
	 * @return the set of the session attributes names
	 */
	public Set<String> getAttributesNames() {
		return this.attributes.keySet();
	}

	/**
	 * Sets the expiration.
	 *
	 * @param expiration
	 *            the expiration to set in millis.
	 */
	protected void setExpiration(long expiration) {
		this.expiration = expiration;
	}

}
