/*
 * Java
 *
 * Copyright 2014-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.websocket;



import ej.bon.Util;
import ej.util.concurrent.SingleThreadExecutor;
import ej.util.message.Level;

/**
 * Section 7.1.1. Close the WebSocket Connection says: "In abnormal cases (such as not having received a TCP Close from the server after a reasonable
 * amount of time) a client MAY initiate the TCP Close."
 * <p>
 * When a websocket client calls {@link WebSocket#close()}, this sends a close frame and the websocket moves to the CLOSING state. Normally, the
 * server will response with an appropriate close frame and will close the TCP connection. This {@link Runnable} is here to manage the case described
 * above: once the client has send the close frame, it starts a new {@link Thread} that runs an {@link OnTimeOutCloser}. This {@link OnTimeOutCloser}
 * will wait for an amount of time before closing the underlying TCP connection.
 * <p>
 * If server responses before the timeout has expired, the {@link Receiver} that manages the incoming data of the websocket will call
 * {@link WebSocket#closeUnderlyingTCPConnection()}, that will check whether the {@link OnTimeOutCloser} thread is running and stops it if necessary
 * and move the websocket to the CLOSED state.
 *
 *
 *
 */
/* default */class OnTimeOutCloser implements Runnable {

	private static final long TIMEOUT_BEFORE_FORCING_TO_CLOSE = 1000;

	private static final Object mutex = new Object();
	private static SingleThreadExecutor EXECUTOR;
	private static int PENDING;

	private final WebSocket ws;
	private boolean closed;
	private long timeout;

	/**
	 * Create a new OnTimeOutCloser that will close the websocket if timeout expires.
	 *
	 * @param ws
	 *            the websocket to close
	 */
	/* default */ OnTimeOutCloser(WebSocket ws) {
		this.ws = ws;
		this.closed = false;
	}

	/* default */boolean isClosed() {
		return closed;
	}

	/* default */void setClosed(boolean closed) {
		this.closed = closed;
	}

	@Override
	public void run() {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.TIMEOUT_START, this);

		try {
			long sleep = timeout - Util.platformTimeMillis();
			if (sleep > 0) {
				Thread.sleep(sleep);
			}

			if (closed) {
				// This is the nominal case: a close frame is sent, this runnable is started, a close frame is received
				// before the timeout is
				// reached, 'closed' is set to 'true' and this runnable is notified.
				Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.TIMEOUT_NORMAL_CLOSE, this);
			} else {
				// The server didn't answer to our closing handshake
				Messages.LOGGER.log(Level.INFO, Messages.CATEGORY, Messages.TIMEOUT, this);
				ws.closeUnderlyingTCPConnection();
				ws.getEndpoint().onClose(ws, new ReasonForClosure(CloseCodes.CONNECTION_CLOSED_ABNORMALLY, ""));
			}
		} catch (InterruptedException ie) {
			// This is an error case, nobody is supposed to interrupt!
			Messages.LOGGER.log(Level.SEVERE, Messages.CATEGORY, Messages.TIMEOUT_INTERRUPTED, this);
			ws.getEndpoint().onError(ws, ie);
			ws.closeUnderlyingTCPConnection();
		}

		synchronized (mutex) {
			PENDING--;
			if (PENDING <= 0) {
				EXECUTOR.shutdown();
				EXECUTOR = null;
			}
		}
	}

	public void schedule() {
		timeout = Util.platformTimeMillis() + TIMEOUT_BEFORE_FORCING_TO_CLOSE;
		synchronized (mutex) {
			if (EXECUTOR == null) {
				EXECUTOR = new SingleThreadExecutor();
				PENDING = 0;
			}
			PENDING++;
		}
		EXECUTOR.execute(this);
	}

}
