/*
 * <Java>
 *
 * Copyright 2015-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.service.loader;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.service.ServiceLoader;
import ej.service.ServiceLoadingException;
import ej.service.ServicePermission;

/**
 * Service loader that creates service instance based on default service implementation class.
 */
public abstract class SimpleServiceLoader implements ServiceLoader {

	/**
	 * The cache of instantiated services.
	 * <p>
	 * Be careful to use it in a synchronized context.
	 *
	 * @see #putService(Class, Object)
	 */
	protected final PackedMap<Class<?>, Object> services;

	/**
	 * Instantiates a new default service loader.
	 */
	public SimpleServiceLoader() {
		this.services = new PackedMap<>();
	}

	@Override
	@Nullable
	public <T> T getService(Class<T> service) {
		checkServicePermission(service, ServicePermission.GET_ACTION);

		synchronized (service) {
			PackedMap<Class<?>, Object> services = this.services;
			@SuppressWarnings("unchecked")
			T cachedService = (T) services.get(service); // cast ensured by cache policy
			if (cachedService != null) {
				return cachedService;
			}
			return createAlternativeImplementation(service);
		}
	}

	/**
	 * Checks that an action on a service is permitted.
	 *
	 * @param service
	 *            the service.
	 * @param action
	 *            the action.
	 * @param <T>
	 *            the service type.
	 */
	protected <T> void checkServicePermission(Class<T> service, String action) {
		SecurityManager securityManager = System.getSecurityManager();
		if (securityManager != null) {
			String className = getClass().getSimpleName();
			securityManager.checkPermission(new ServicePermission(className, service, action));
		}
	}

	/**
	 * Creates an implementation for the given service and save it in the cache.
	 *
	 * @param service
	 *            the service.
	 * @param <T>
	 *            the service type.
	 * @return a new instance of the given service or <code>null</code>.
	 * @throws ServiceLoadingException
	 *             if an error occurs while instantiating the service instance.
	 */
	@Nullable
	protected <T> T createAlternativeImplementation(Class<T> service) throws ServiceLoadingException {
		return null;
	}

	/**
	 * Put a service in the services map in a synchronized context.
	 *
	 * @param service
	 *            the service.
	 * @param instance
	 *            the service instance.
	 * @param <T>
	 *            the service type.
	 */
	protected <T> void putService(Class<T> service, T instance) {
		synchronized (this.services) {
			this.services.put(service, instance);
		}
	}

}
