/*
 * Copyright 2014-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.websocket.http;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;


/**
 * HTTP Response as described by RFC 2616, section 6. Response.
 *
 *
 *
 */
public class HttpResponse {

	private static final int ORIGINAL_LINE_BUFFER_SIZE = 128;

	private static final String CRLF = "\r\n";

	private final StatusLine statusLine;
	private final List<Header> allHeaders;

	/**
	 * Read the input stream until a complete HTTP response is formed.
	 * <p>
	 * WARNING: this method won't return until the response is complete!
	 *
	 * @param is
	 *            the stream to read the data from
	 * @throws IOException
	 *             if an error occurs on the stream
	 */
	public HttpResponse(InputStream is) throws IOException {
		// First line is status line
		String line = getNextLine(is);
		statusLine = new StatusLine(line);

		// Next lines are header fields
		allHeaders = new ArrayList<Header>();

		while (true) {
			line = getNextLine(is);
			if (CRLF.equals(line)) {
				break; // response has been completely received
			}

			Header header = new Header(line);
			allHeaders.add(header);
		}
	}

	/**
	 * @return the status line of the response
	 */
	public StatusLine getStatusLine() {
		return statusLine;
	}

	/**
	 * @param name
	 *            the name of the desired header field
	 * @return the corresponding header or null is the response doesn't contain such a header
	 */
	public Header getHeader(String name) {
		Header header = null;

		Iterator<Header> iterator = allHeaders.iterator();
		while (iterator.hasNext()) {
			Header current = iterator.next();
			if (current.getName().compareToIgnoreCase(name) == 0) {
				header = current;
				break; // stop the loop, we found the header we were looking for
			}
		}

		return header;
		// TODO HttpMessage from Apache shows that there may several Headers with the same name
	}

	/**
	 * @return all the header fields of the response
	 */
	public List<Header> getAllHeaders() {
		return new ArrayList<>(allHeaders);
	}

	/**
	 * Helpful method to read a new line from the stream. A line is complete one "\r\n" (CRLF) is received.
	 *
	 * @param is
	 *            the input stream to read data from
	 * @return a new line ending with trailing CRLF included
	 * @throws IOException
	 *             if an error occurs on the stream
	 */
	private String getNextLine(InputStream is) throws IOException {
		char[] buffer = new char[ORIGINAL_LINE_BUFFER_SIZE];
		int i = 0;

		int previous = 0;
		int current = 0;
		while (true) {
			// Get next character from the stream
			previous = current;
			current = is.read();

			if (current == -1) {
				// End of stream reached
				throw new IOException();
			}

			// Add it into the buffer
			buffer[i] = (char) current;
			i++;

			// This buffer may be full and need to be resized
			if(i == buffer.length) {
				char[] temp = new char[buffer.length * 2];
				System.arraycopy(buffer, 0, temp, 0, i);
				buffer = temp;
			}

			// Check if the line is complete
			if (previous == '\r' && current == '\n') {
				break; // Line is complete
			}
		}

		return String.valueOf(buffer, 0, i);
	}

	@Override
	public String toString() {
		return "HttpResponse [" + statusLine + "\n" + allHeaders + "]";
	}
}
